/*
 *	$Id: redhatclient.cpp,v 1.6 2002/12/17 19:13:10 than Exp $
 *
 *	KDE2 Default KWin client
 *
 *	Copyright (C) 1999, 2001 Daniel Duley <mosfet@kde.org>
 *	Matthias Ettrich <ettrich@kde.org>
 *	Karol Szwed <gallium@kde.org>
 *
 *	Draws mini titlebars for tool windows.
 *	Many features are now customizable.
 */

#include <kconfig.h>
#include <kglobal.h>
#include <kpixmapeffect.h>
#include <kimageeffect.h>
#include <kdrawutil.h>
#include <klocale.h>
#include <qlayout.h>
#include <qdrawutil.h>
#include <qbitmap.h>
#include <kwin/workspace.h>
#include <kwin/options.h>
#include "redhatclient.h"
#include <qimage.h>

#define BASE_BUTTON_SIZE  17
#define BORDER_WIDTH      6
#define CORNER_RADIUS     12

#define BUTTON_DIAM 12
#define TOP_GRABBAR_WIDTH 2
#define BOTTOM_CORNER 5


using namespace KWinInternal;

namespace
{

  /* XPM */
  static const char * bottom_right_xpm[] = {
    "22 22 5 1",
    " 	c None",
    ".	c #727272",
    "+	c #929292",
    "@	c #000000",
    "#	c #AEAEAE",
    "                .+..@@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "                .#++.@",
    "+...............#+++.@",
    "################+++.@ ",
    "#++++++++++++++++++.@ ",
    "#++++++++++++++++..@  ",
    "+................@@   ",
    "@@@@@@@@@@@@@@@@@     "};

  /* XPM */
  static const char * bottom_left_xpm[] = {
    "22 22 5 1",
    " 	c None",
    ".	c #000000",
    "+	c #929292",
    "@	c #727272",
    "#	c #AEAEAE",
    ".+@@@@                ",
    ".####@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#+++@                ",
    ".#++++@@@@@@@@@@@@@@@@",
    " .#++++##############+",
    " .#++++++++++++++++++@",
    "  .@@++++++++++++++++@",
    "   ..@@@@@@@@@@@@@@@@@",
    "     ................."};

  static unsigned char iconify_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xf8, 0x07, 0xfc, 0x0f, 0xfc, 0x0f, 0xf8, 0x07,
    0x00, 0x00, 0x00, 0x00 };

  static unsigned char close_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x18, 0x06, 0x3c, 0x0f, 0xfc, 0x0f, 0xf8, 0x07,
    0xf0, 0x03, 0xf0, 0x03, 0xf8, 0x07, 0xfc, 0x0f, 0x3c, 0x0f, 0x18, 0x06,
    0x00, 0x00, 0x00, 0x00 };

  static unsigned char maximize_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0xf8, 0x07, 0xfc, 0x0f, 0xfc, 0x0f, 0xfc, 0x0f,
    0x04, 0x08, 0x04, 0x08, 0x04, 0x08, 0x04, 0x08, 0x04, 0x08, 0xf8, 0x07,
    0x00, 0x00, 0x00, 0x00 };

  static unsigned char minmax_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0xe0, 0x07, 0xf0, 0x0f, 0x00, 0x08, 0xfc, 0x0b,
    0xfc, 0x0b, 0xfc, 0x0b, 0x04, 0x0a, 0x04, 0x02, 0x04, 0x02, 0xfc, 0x03,
    0x00, 0x00, 0x00, 0x00 };

  static unsigned char question_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0xf0, 0x00, 0x98, 0x01, 0x80, 0x01, 0xc0, 0x00,
    0x60, 0x00, 0x00, 0x00, 0x60, 0x00, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00 };

  static unsigned char menu_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x0c, 0x1e, 0x1e, 0x3e, 0x1f,
    0xfc, 0x0f, 0xf8, 0x07, 0xf0, 0x03, 0xe0, 0x01, 0xc0, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00 };

  static unsigned char pindown_white_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x80, 0x1f, 0xa0, 0x03,
    0xb0, 0x01, 0x30, 0x01, 0xf0, 0x00, 0x70, 0x00, 0x20, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pindown_gray_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c,
    0x00, 0x0e, 0x00, 0x06, 0x00, 0x00, 0x80, 0x07, 0xc0, 0x03, 0xe0, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pindown_dgray_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xc0, 0x10, 0x70, 0x20, 0x50, 0x20,
    0x48, 0x30, 0xc8, 0x38, 0x08, 0x1f, 0x08, 0x18, 0x10, 0x1c, 0x10, 0x0e,
    0xe0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pindown_mask_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xc0, 0x1f, 0xf0, 0x3f, 0xf0, 0x3f,
    0xf8, 0x3f, 0xf8, 0x3f, 0xf8, 0x1f, 0xf8, 0x1f, 0xf0, 0x1f, 0xf0, 0x0f,
    0xe0, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pinup_white_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x80, 0x11,
    0x3f, 0x15, 0x00, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pinup_gray_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x80, 0x0a, 0xbf, 0x0a, 0x80, 0x15, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pinup_dgray_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x20, 0x40, 0x31, 0x40, 0x2e,
    0x40, 0x20, 0x40, 0x20, 0x7f, 0x2a, 0x40, 0x3f, 0xc0, 0x31, 0xc0, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  static unsigned char pinup_mask_bits[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0x20, 0xc0, 0x31, 0xc0, 0x3f,
    0xff, 0x3f, 0xff, 0x3f, 0xff, 0x3f, 0xc0, 0x3f, 0xc0, 0x31, 0xc0, 0x20,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

  // ===========================================================================

  KPixmap* titlePix 			= NULL;
  KPixmap* titleBuffer 		= NULL;
  KPixmap* aUpperGradient 	= NULL;
  KPixmap* iUpperGradient 	= NULL;

  KPixmap* pinDownPix			= NULL;
  KPixmap* pinUpPix			= NULL;
  KPixmap* ipinDownPix		= NULL;
  KPixmap* ipinUpPix			= NULL;

  KPixmap* btnUpPix		= NULL;
  KPixmap* btnDownPix	= NULL;
  KPixmap* ibtnUpPix		= NULL;
  KPixmap* ibtnDownPix	= NULL;


  QPixmap* bottomLeftPix		= NULL;
  QPixmap* bottomRightPix		= NULL;
  QPixmap* abottomLeftPix		= NULL;
  QPixmap* abottomRightPix	= NULL;


  RedHatHandler* clientHandler;
  bool RedHat_initialized = false;
  bool useGradients;
  bool showGrabBar;
  bool showTitleBarStipple;
  bool largeToolButtons;
  int	 toolTitleHeight;
  int	 normalTitleHeight;

}

// ===========================================================================

RedHatHandler::RedHatHandler()
{
  readConfig();
  createPixmaps();
  RedHat_initialized = true;
}


RedHatHandler::~RedHatHandler()
{
  RedHat_initialized = false;
  freePixmaps();
}


void RedHatHandler::reset()
{
  RedHat_initialized = false;
  freePixmaps();
  readConfig();
  createPixmaps();
  RedHat_initialized = true;
  Workspace::self()->slotResetAllClientsDelayed();
}


void RedHatHandler::readConfig()
{
  KConfig* conf = KGlobal::config();
  conf->setGroup("RedHat");

  showGrabBar 		= conf->readBoolEntry("ShowGrabBar", true);
  showTitleBarStipple   = conf->readBoolEntry("ShowTitleBarStipple", true);
  useGradients 		= conf->readBoolEntry("UseGradients", true);
  int size 		= conf->readNumEntry("TitleBarSize", 0);

  if (size < 0) size = 0;
  if (size > 2) size = 2;

  normalTitleHeight 	= BASE_BUTTON_SIZE + (4*size);
  toolTitleHeight 	= normalTitleHeight - 4;
  largeToolButtons 	= (toolTitleHeight >= 16) ? true : false;
}


// This paints the button pixmaps upon loading the style.
void RedHatHandler::createPixmaps()
{
  // Make the titlebar stipple optional
  if (showTitleBarStipple)
    {
      QPainter p;
      QPainter maskPainter;
      int x, y;
      titlePix = new KPixmap();
      titlePix->resize(132, normalTitleHeight+2);
      QBitmap mask(132, normalTitleHeight+2);


      mask.fill(Qt::color0);

      p.begin(titlePix);
      maskPainter.begin(&mask);
      maskPainter.setPen(Qt::color1);

      QColor lighterColor (options->color(Options::TitleBar, true).light (150));
      int h, s, v;
      lighterColor.hsv (&h, &s, &v);
      s /= 2.0;
      s = (s > 255) ? 255 : (int) s;
      QColor satColor(h, s, v, QColor::Hsv);

      KPixmapEffect::gradient(*titlePix,
			      satColor,
			      satColor.dark(150),
			      KPixmapEffect::VerticalGradient);

      for(y = 0; y < (normalTitleHeight+2); y++) {
	for(x = (3 - y) % 4; x < 132; x += 4) {
	  maskPainter.drawPoint(x, y);
	}
      }

      maskPainter.end();
      p.end();
      titlePix->setMask(mask);
    } else
      titlePix = NULL;


  // Create titlebar gradient images if required
  aUpperGradient = NULL;
  iUpperGradient = NULL;

  // Set the sticky pin pixmaps;
  QColorGroup g;
  QPainter p;

  // Active pins
  g = options->colorGroup( Options::ButtonBg, true );
  pinUpPix = new KPixmap();
  pinUpPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  p.begin( pinUpPix );
  kColorBitmaps( &p, g, 0, 0, BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, true, pinup_white_bits,
		 pinup_gray_bits, NULL, NULL, pinup_dgray_bits, NULL );
  p.end();
  pinUpPix->setMask( QBitmap(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, pinup_mask_bits, true) );

  pinDownPix = new KPixmap();
  pinDownPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  p.begin( pinDownPix );
  kColorBitmaps( &p, g, 0, 0, BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, true, pindown_white_bits,
		 pindown_gray_bits, NULL, NULL, pindown_dgray_bits, NULL );
  p.end();
  pinDownPix->setMask( QBitmap(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, pindown_mask_bits, true) );

  // Inactive pins
  g = options->colorGroup( Options::ButtonBg, false );
  ipinUpPix = new KPixmap();
  ipinUpPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  p.begin( ipinUpPix );
  kColorBitmaps( &p, g, 0, 0, BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, true, pinup_white_bits,
		 pinup_gray_bits, NULL, NULL, pinup_dgray_bits, NULL );
  p.end();
  ipinUpPix->setMask( QBitmap(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, pinup_mask_bits, true) );

  ipinDownPix = new KPixmap();
  ipinDownPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  p.begin( ipinDownPix );
  kColorBitmaps( &p, g, 0, 0, BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, true, pindown_white_bits,
		 pindown_gray_bits, NULL, NULL, pindown_dgray_bits, NULL );
  p.end();
  ipinDownPix->setMask( QBitmap(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE, pindown_mask_bits, true) );

  // Create a title buffer for flicker-free painting
  titleBuffer = new KPixmap();

  // Cache all possible button states

  btnUpPix 	= new KPixmap();
  btnUpPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  btnDownPix = new KPixmap();
  btnDownPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  ibtnUpPix 	= new KPixmap();
  ibtnUpPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);
  ibtnDownPix = new KPixmap();
  ibtnDownPix->resize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);

  // Draw the button state pixmaps
  g = options->colorGroup( Options::ButtonBg, true );
  drawButtonBackground( btnUpPix, g, false, true );
  drawButtonBackground( btnDownPix, g, true, true );

  g = options->colorGroup( Options::ButtonBg, false );
  drawButtonBackground( ibtnUpPix, g, false, false );
  drawButtonBackground( ibtnDownPix, g, true, false );

  QImage bottomleft(bottom_left_xpm);
  QImage bottomright(bottom_right_xpm);
  QImage abottomleft(bottom_left_xpm);
  QImage abottomright(bottom_right_xpm);
  recolor(bottomleft, options->color( Options::TitleBar, false ).light(95));
  recolor(bottomright, options->color( Options::TitleBar, false ).light(95));
  recolor(abottomleft, options->color( Options::TitleBar, true ).light(135));
  recolor(abottomright, options->color( Options::TitleBar, true ).light(135));

  bottomLeftPix 	= new QPixmap();
  bottomRightPix	= new QPixmap();
  abottomLeftPix	= new QPixmap();	
  abottomRightPix	= new QPixmap();
  bottomLeftPix->convertFromImage(bottomleft);
  bottomRightPix->convertFromImage(bottomright);
  abottomLeftPix->convertFromImage(abottomleft);
  abottomRightPix->convertFromImage(abottomright);
}

// This is the recoloring method from the Keramik widget style,
// copyright (c) 2002 Malte Starostik <malte@kde.org>.
// Modified to work with 8bpp images.
void RedHatHandler::recolor( QImage &img, const QColor& color )
{
  int hue = -1, sat = 0, val = 228;
  if ( color.isValid() ) color.hsv( &hue, &sat, &val );
  register int pixels = (img.depth() > 8 ? img.width() * img.height() : img.numColors());
  register Q_UINT32* data = ( img.depth() > 8 ? reinterpret_cast< Q_UINT32* >( img.bits() ) :
			      reinterpret_cast< Q_UINT32* >( img.colorTable() ) );
	
  for ( int i = 0; i < pixels; i++ )
    {
      QColor c( *data );
      int h, s, v;
      c.hsv( &h, &s, &v );
      h = hue;
      s = sat;
      v = v * val / 145;
      c.setHsv( h, QMIN( s, 255 ), QMIN( v, 255 ) );
      *data = ( c.rgb() & RGB_MASK ) | ( *data & ~RGB_MASK );
      data++;
    }
}

void RedHatHandler::freePixmaps()
{
  // Free button pixmaps
  if (btnUpPix)
    delete btnUpPix;
  if(btnDownPix)
    delete btnDownPix;
  if (ibtnUpPix)
    delete ibtnUpPix;
  if (ibtnDownPix)
    delete ibtnDownPix;

  // Title images
  if (titleBuffer)
    delete titleBuffer;
  if (titlePix)
    delete titlePix;
  if (aUpperGradient)
    delete aUpperGradient;
  if (iUpperGradient)
    delete iUpperGradient;

  // Sticky pin images
  if (pinUpPix)
    delete pinUpPix;
  if (ipinUpPix)
    delete ipinUpPix;
  if (pinDownPix)
    delete pinDownPix;
  if (ipinDownPix)
    delete ipinDownPix;
}


void RedHatHandler::drawButtonBackground(KPixmap *pix, 
					 const QColorGroup &g, bool sunken, bool active)
{
  QPainter p;

  bool highcolor = useGradients && (QPixmap::defaultDepth() > 8);
  QColor c = g.background();

  // Fill the background with a gradient if possible
  if (highcolor)
    {
      if (active)
	{
	  KPixmapEffect::gradient(*pix, c, Qt::white,
				  KPixmapEffect::DiagonalGradient);
	}
      else
	{
	  QColor inactiveTitleColor1(options->color(Options::TitleBar,    false));
	  QColor inactiveTitleColor2(options->color(Options::TitleBlend,  false));
	  KPixmapEffect::gradient(*pix,
				  inactiveTitleColor2,
				  inactiveTitleColor1,
				  KPixmapEffect::VerticalGradient);
	  
	}
    }
  else
    pix->fill(c);

  p.begin(pix);
  // outer frame
  //      p.setPen(g.mid());
  //      p.drawLine(0, 0, x2, 0);
  //      p.drawLine(0, 0, 0, y2);
  //      p.setPen(g.light());
  //      p.drawLine(x2, 0, x2, y2);
  //      p.drawLine(0, x2, y2, x2);
  //      p.setPen(g.dark());
  //      p.drawRect(1, 1, w-2, h-2);

  p.setPen(sunken ? g.dark() : g.mid());
  //p.drawLine(0, 0, w, 0);
  //     p.drawLine(x2-1, 0, x2-1, h);
  //p.drawLine(0, 0, 0, h);
  //p.drawLine(0, w, h, w);
}


// ===========================================================================

RedHatButton::RedHatButton(Client *parent, const char *name, 
			   bool largeButton, int bpos, bool isStickyButton,
			   const unsigned char *bitmap, const QString& tip )
  : KWinButton(parent, name, tip)
{
  setBackgroundMode( QWidget::NoBackground );
  setToggleButton( isStickyButton );

  isMouseOver = false;
  deco 		= NULL;
  large 		= largeButton;
  isSticky 	= isStickyButton;
  client 		= parent;

  pos = bpos;

  setFixedSize(BASE_BUTTON_SIZE, BASE_BUTTON_SIZE);

  if (bitmap)
    setBitmap(bitmap);
}


RedHatButton::~RedHatButton()
{
  if (deco)
    delete deco;
}


QSize RedHatButton::sizeHint() const
{
  return( QSize(BASE_BUTTON_SIZE,BASE_BUTTON_SIZE ));
}


void RedHatButton::resizeEvent( QResizeEvent* e)
{
  doShape();
  KWinButton::resizeEvent(e);
}

void RedHatButton::showEvent(QShowEvent *ev)
{
  doShape();
  KWinButton::showEvent(ev);
}

void RedHatButton::doShape()
{
  // Obtain widget bounds.
  int w  = rect() .width();
  int h  = rect().height();
  int r = BUTTON_DIAM / 2;
  int dm = BUTTON_DIAM;
  QBitmap mask(size(), true);

  QPainter p3(&mask);
  QBrush blackbr(Qt::color1);
  p3.fillRect(0,0,w,h,blackbr);

  p3.setPen(Qt::color1);
  p3.setBrush(Qt::color1);
  if (pos == ButtonLeft) {
    p3.eraseRect(0, -TOP_GRABBAR_WIDTH, r, r);
    p3.drawPie(0, -TOP_GRABBAR_WIDTH, dm-1, dm-1, 90*16, 90*16);
    p3.drawArc(0, -TOP_GRABBAR_WIDTH, dm-1, dm-1, 90*16, 90*16);
  } else if (pos == ButtonRight) {
    p3.eraseRect(w-r , -TOP_GRABBAR_WIDTH, r,r);
    p3.drawPie(w-dm, -TOP_GRABBAR_WIDTH, dm-1, dm-1, 0*16, 90*16);
    p3.drawArc(w-dm, -TOP_GRABBAR_WIDTH, dm-1, dm-1, 0*16, 90*16);
  }
  p3.end();
  setMask(mask);
}

void RedHatButton::setBitmap(const unsigned char *bitmap)
{
  if (deco)
    delete deco;

  deco = new QBitmap(14, 14, bitmap, true);
  deco->setMask( *deco );
  repaint( false );
}


void RedHatButton::drawButton(QPainter *p)
{
  if (!RedHat_initialized)
    return;

  if (deco)
    {
      // Fill the button background with an appropriate button image
      KPixmap btnbg;

      if (isDown())
	btnbg = client->isActive() ? *btnDownPix : *ibtnDownPix;
      else
	btnbg = client->isActive() ? *btnUpPix : *ibtnUpPix;

      // Scale the background image if required
      // This is slow, but we assume this isn't done too often
      if ( !large )
	{
	  btnbg.detach();
	  btnbg.convertFromImage(btnbg.convertToImage().smoothScale(14, 14));
	}

      p->drawPixmap( 0, 0, btnbg );

    }

  // If we have a decoration bitmap, then draw that
  // otherwise we paint a menu button (with mini icon), or a sticky button.
  if( deco )
    {
      // Select the appropriate button decoration color
      bool darkDeco = qGray( options->color( 
					    Options::ButtonBg,
					    client->isActive()).rgb() ) > 127;

      QColor bgc = options->color(Options::TitleBar, client->isActive());

      if (isMouseOver)
	p->setPen( darkDeco ? bgc.dark(120) : bgc.light(120) );
      else
	p->setPen( darkDeco ? bgc.dark(150) : bgc.light(150) );

      int xOff = (width()-14)/2;
      int yOff = (height()-14)/2;
      p->drawPixmap(isDown() ? xOff+1: xOff, isDown() ? yOff+1 : yOff, *deco);

    }
  else
    {
      KPixmap btnpix;

      if (isSticky)
	{
	  if (client->isActive())
	    btnpix = isOn() ? *pinDownPix : *pinUpPix;
	  else
	    btnpix = isOn() ? *ipinDownPix : *ipinUpPix;
	}
      else
	{
	  btnpix = client->miniIcon();
	}
      
      // Intensify the image if required
      if (isMouseOver)
	{
	  btnpix = KPixmapEffect::intensity(btnpix, 0.8);
	}
      
      // Smooth scale the pixmap for small titlebars
      // This is slow, but we assume this isn't done too often
      if ( !large )
	btnpix.convertFromImage(btnpix.convertToImage().smoothScale(14, 14));
      
      p->drawPixmap( 0, 0, btnpix );
    }

  QColorGroup g;
  g = options->colorGroup( Options::ButtonBg, false );
  p->setPen(g.dark());
  //     if (pos != ButtonRight)
  //       p->drawLine(width()-1, 0, width()-1, height());

  //     int dm = BUTTON_DIAM;
  //     p->setPen(Qt::black);
  //     p->setBrush(Qt::black);
  //     if (pos == ButtonLeft) {
  //       p->drawArc(-1, -TOP_GRABBAR_WIDTH, dm, dm, 90*16, 90*16);
  //     } else if (pos == ButtonRight) {
  //       p->drawArc(width()-dm+1, -TOP_GRABBAR_WIDTH, dm, dm, 0*16, 90*16);
  //     }
}


// Make the protected member public
void RedHatButton::turnOn( bool isOn )
{
  if ( isToggleButton() )
    setOn( isOn );
}


void RedHatButton::enterEvent(QEvent *e) 
{ 
  isMouseOver=true;
  repaint(false); 
  KWinButton::enterEvent(e);
}


void RedHatButton::leaveEvent(QEvent *e)
{ 
  isMouseOver=false;
  repaint(false); 
  KWinButton::leaveEvent(e);
}


void RedHatButton::mousePressEvent( QMouseEvent* e )
{
  last_button = e->button();
  QMouseEvent me( e->type(), e->pos(), e->globalPos(),
                  LeftButton, e->state() );

  KWinButton::mousePressEvent( &me );
}


void RedHatButton::mouseReleaseEvent( QMouseEvent* e )
{
  last_button = e->button();
  QMouseEvent me( e->type(), e->pos(), e->globalPos(),
                  LeftButton, e->state() );

  KWinButton::mouseReleaseEvent( &me );
}


// ===========================================================================

RedHatClient::RedHatClient( Workspace *ws, WId w, QWidget *parent,
			    const char *name )
  : Client( ws, w, parent, name, WResizeNoErase | WStaticContents |
	    WRepaintNoErase )
{
  // No flicker thanks
  setBackgroundMode( QWidget::NoBackground );

  // Set button pointers to NULL so we can track things
  for(int i=0; i < RedHatClient::BtnCount; i++)
    button[i] = NULL;

  // Finally, toolWindows look small
  if ( isTool() ) {
    titleHeight  = toolTitleHeight;
    largeButtons = largeToolButtons;
  }
  else {
    titleHeight  = normalTitleHeight;
    largeButtons = true;
  }

  // Pack the windowWrapper() window within a grid
  QVBoxLayout* g = new QVBoxLayout(this);
  g->setResizeMode(QLayout::FreeResize);
  g->addSpacing(TOP_GRABBAR_WIDTH);       // Top grab bar

  // Pack the titlebar HBox with items
  hb = new QHBoxLayout();
  hb->setSpacing(0);
  hb->setMargin(0);
  hb->setResizeMode( QLayout::FreeResize );

  hb->addSpacing(2);
  addClientButtons( options->titleButtonsLeft(), true );
  titlebar = new QSpacerItem( 10, titleHeight,
			      QSizePolicy::Expanding, QSizePolicy::Minimum );
  hb->addItem(titlebar);

  hb->addSpacing(2);
  addClientButtons( options->titleButtonsRight(), false );

  hb->addSpacing(2);

  g->addLayout( hb );
  g->addSpacing(1);       // line under titlebar

  // Add the middle section
  hb = new QHBoxLayout();
  hb->addSpacing(BORDER_WIDTH);
  hb->addWidget(windowWrapper(), 0);
  hb->addSpacing(BORDER_WIDTH);
  g->addLayout( hb );

  // Determine the size of the lower grab bar
  if ( showGrabBar && (!isTool()) )
    g->addSpacing(BORDER_WIDTH);   // bottom handles
  else
    g->addSpacing(4);   // bottom handles
}


void RedHatClient::addClientButtons( const QString& s, bool isLeft )
{
  int pos;
  // Make sure we place the spacing between the buttons
  bool first_button = true;
  RedHatButton* last_button = NULL;
  if (s.length() > 0) {
    for(unsigned int i = 0; i < s.length(); i++) {
      if (i == 0 && isLeft)
	pos = ButtonLeft;
      else pos = ButtonMid;
      if (! first_button)
	hb->addSpacing(2);
      else
	first_button = false;
      switch( s[i].latin1() )
	{
	  // Menu button
	case 'M':
	  if (!button[BtnMenu])
	    {
	      button[BtnMenu] = new RedHatButton(this, "menu",
						 largeButtons, pos, false, menu_bits, i18n("Menu"));
	      connect( button[BtnMenu], SIGNAL(pressed()),
		       this, SLOT(menuButtonPressed()) );
	      hb->addWidget( button[BtnMenu] );
	      last_button = button[BtnMenu];
	    }
	  break;

	  // Sticky button
	case 'S':
	  /*
	  if (!button[BtnSticky])
	    {
	      button[BtnSticky] = new RedHatButton(this, "sticky", 
						   largeButtons, pos, true, NULL, i18n("Sticky"));
	      button[BtnSticky]->turnOn( isSticky() );
	      connect( button[BtnSticky], SIGNAL(clicked()), 
		       this, SLOT(toggleSticky()) );
	      hb->addWidget( button[BtnSticky] );
	      last_button = button[BtnSticky];
	      }*/
	  break;

	  // Help button
	case 'H':
	  if( providesContextHelp() && (!button[BtnHelp]) )
	    {
	      button[BtnHelp] = new RedHatButton(this, "help",
						 largeButtons, pos, true, question_bits,
						 i18n("Help"));
	      connect( button[BtnHelp], SIGNAL( clicked() ),
		       this, SLOT( contextHelp() ));
	      hb->addWidget( button[BtnHelp] );
	      last_button = button[BtnHelp];
	    }
	  break;

	  // Minimize button
	case 'I':
	  if ( (!button[BtnIconify]) && isMinimizable())
	    {
	      button[BtnIconify] = new RedHatButton(this, "iconify",
						    largeButtons, pos, false, iconify_bits,
						    i18n("Minimize"));
	      connect( button[BtnIconify], SIGNAL( clicked()),
		       this, SLOT(iconify()) );
	      hb->addWidget( button[BtnIconify] );
	      last_button = button[BtnIconify];
	    }
	  break;

	  // Maximize button
	case 'A':
	  if ( (!button[BtnMax]) && isMaximizable())
	    {
	      button[BtnMax]  = new RedHatButton(this, "maximize",
						 largeButtons, pos, false, maximize_bits,
						 i18n("Maximize"));
	      connect( button[BtnMax], SIGNAL( clicked()),
		       this, SLOT(slotMaximize()) );
	      hb->addWidget( button[BtnMax] );
	      last_button = button[BtnMax];
	    }
	  break;

	  // Close button
	case 'X':
	  if (!button[BtnClose])
	    {
	      button[BtnClose] = new RedHatButton(this, "close",
						  largeButtons, pos, false, close_bits,
						  i18n("Close"));
	      connect( button[BtnClose], SIGNAL( clicked()),
		       this, SLOT(closeWindow()) );
	      hb->addWidget( button[BtnClose] );
	      last_button = button[BtnClose];
	    }
	  break;

	  // Spacer item (only for non-tool windows)
	case '_':
	  if ( !isTool() )
	    hb->addSpacing(2);
	}
    }

    if(not isLeft)
      last_button->pos = ButtonRight;
    else
      last_button->pos = LeftButtonRight;
  }
}


void RedHatClient::iconChange()
{
  if (button[BtnMenu] && button[BtnMenu]->isVisible())
    button[BtnMenu]->repaint(false);
}

void RedHatClient::stickyChange(bool on)
{
  if (button[BtnSticky]) {
    button[BtnSticky]->turnOn(on);
    button[BtnSticky]->repaint(false);
    button[BtnSticky]->setTipText(on ? i18n("Un-Sticky") : i18n("Sticky"));
  }
}

void RedHatClient::slotMaximize()
{
  if ( button[BtnMax]->last_button == MidButton )
    maximize( MaximizeVertical );
  else if ( button[BtnMax]->last_button == RightButton )
    maximize( MaximizeHorizontal );
  else
    maximize();
}


void RedHatClient::resizeEvent( QResizeEvent* e)
{
  Client::resizeEvent( e );
  doShape();
  calcHiddenButtons();
  
  if (isVisibleToTLW())
    {
      update(rect());
      int dx = 0;
      int dy = 0;

      if ( e->oldSize().width() != width() )
	dx = 32 + QABS( e->oldSize().width() -  width() );

      if ( e->oldSize().height() != height() )
	dy = 8 + QABS( e->oldSize().height() -  height() );

      if ( dy )
	update( 0, height() - dy + 1, width(), dy );

      if ( dx )
	{
	  update( width() - dx + 1, 0, dx, height() );
	  update( QRect( QPoint(4,4), titlebar->geometry().bottomLeft() - 
			 QPoint(1,0) ) );
	  update( QRect( titlebar->geometry().topRight(), QPoint(width() - 4,
								 titlebar->geometry().bottom()) ) );
	  // Titlebar needs no paint event
	  QApplication::postEvent( this, new QPaintEvent(titlebar->geometry(),
							 FALSE) );
	}
    }
}


void RedHatClient::captionChange( const QString& )
{
  repaint( titlebar->geometry(), false );
}


void RedHatClient::paintEvent( QPaintEvent* )
{
  if (!RedHat_initialized)
    return;

  QColorGroup g;
  int offset;

  bool drawLeftDivider = true; 
  bool drawRightDivider = true; 
  QPainter p(this);


  // Obtain widget bounds.
  QRect r(rect());
  int x = r.x();
  int y = r.y();
  int x2 = x + r.width() - 1;
  int y2 = y + r.height() - 1;
  int w  = r.width();
  int h  = r.height();
  g = options->colorGroup(Options::Frame, isActive());

  // Determine where to place the extended left titlebar
  int leftFrameStart = (h > 42) ? y+titleHeight+26: y+titleHeight;

  // Determine where to make the titlebar color transition
  r = titlebar->geometry();
  int rightOffset = r.x()+r.width()+1;

  QColor c2 = options->color(Options::Frame, isActive() );
  // Fill with frame color behind RHS buttons
  ///  p.fillRect( x, y+2, x2, titleHeight+1, c2);

  // Create a disposable pixmap buffer for the titlebar
  // very early before drawing begins so there is no lag
  // during painting pixels.
  titleBuffer->resize( w, titleHeight + TOP_GRABBAR_WIDTH );


  // Draw the title bar.
  r = titlebar->geometry();

  // Obtain titlebar blend colours
  QColor c1 = options->color(Options::TitleBar, isActive() );


  QPainter p2( titleBuffer, this );
  QColor activeTitleColor1(options->color(Options::TitleBar,      true));
  QColor activeTitleColor2(options->color(Options::TitleBlend,    true));

  QColor inactiveTitleColor1(options->color(Options::TitleBar,    false));
  QColor inactiveTitleColor2(options->color(Options::TitleBlend,  false));
  bool highcolor = useGradients && (QPixmap::defaultDepth() > 8);

  if (highcolor)
    {
      static QSize oldsize(0,0);
      
      QSize titleBufferSize(w, titleHeight + TOP_GRABBAR_WIDTH);
      if(oldsize != titleBufferSize)
	{
	  oldsize = titleBufferSize;
	  
	  if(aUpperGradient)
	    {
	      delete aUpperGradient;
	      aUpperGradient = NULL;
	    }

	  if(iUpperGradient)
	    {
	      delete iUpperGradient;
	      iUpperGradient = NULL;
	    }
	      
	  // Create the titlebar gradients
	  if (activeTitleColor1 != activeTitleColor2)
	    {
	      aUpperGradient = new KPixmap(oldsize);
	      KPixmapEffect::gradient(*aUpperGradient,
				      activeTitleColor2,
				      activeTitleColor1,
				      KPixmapEffect::VerticalGradient);
	    }

	  if (inactiveTitleColor1 != inactiveTitleColor2)
	    {
	      iUpperGradient = new KPixmap(oldsize);
	      KPixmapEffect::gradient(*iUpperGradient,
				      inactiveTitleColor2,
				      inactiveTitleColor1,
				      KPixmapEffect::VerticalGradient);
	    }
	}
    }
  
  KPixmap* upperGradient = isActive() ? aUpperGradient : iUpperGradient;
  // Draw the titlebar gradient

  if (upperGradient)
    p2.drawPixmap(0, TOP_GRABBAR_WIDTH, *upperGradient);
  else
    p2.fillRect(0, TOP_GRABBAR_WIDTH, w, titleHeight, c1);

  QFont fnt = options->font(true);

  if ( isTool() )
    fnt.setPointSize( fnt.pointSize()-2 );  // Shrink font by 2pt

  p2.setFont( fnt );
  // Draw the titlebar stipple if active and available
  if (isActive() && titlePix)
    {
      QFontMetrics fm(fnt);
      int captionWidth = fm.width(caption());
      p2.drawTiledPixmap( r.x() + captionWidth - 25, TOP_GRABBAR_WIDTH,
			  r.width(), titleHeight+1, *titlePix );
    }

  if (isActive()) {
    p2.setPen( options->color(Options::TitleBlend, isActive()).dark());
    p2.drawText(r.x() + 2 + 1, TOP_GRABBAR_WIDTH + 1,
		r.width(), r.height(),
		AlignLeft | AlignVCenter, caption() );
  }
  p2.setPen( options->color(Options::Font, isActive()) );
  p2.drawText(r.x() + 2, TOP_GRABBAR_WIDTH,
	      r.width(), r.height(),
	      AlignLeft | AlignVCenter, caption() );


  // *** Main Title Bar background area


  p2.setPen(Qt::white);
  p2.drawLine(x + 1, y + 1, x2 - 1, y + 1);
  // This is kind of broken...
  // We fill in the inner part of the circle here.  This is dependent on BUTTON_DIAM
  p2.drawLine(x + 1, y + 1, x + 1, y + TOP_GRABBAR_WIDTH + titleHeight);
  p2.drawLine(x + 2, y + 2, x + 3, y + 2);
  p2.drawLine(x + 2, y + 2, x + 2, y + 3);
  p2.drawLine(x + w - 2 , y + 1, x + w - 2, y + TOP_GRABBAR_WIDTH + titleHeight);
  p2.drawLine(x + w - 3, y + 2, x + w - 3, y + 5);
  p2.drawLine(x + w - 4, y + 2, x + w - 3, y + 2);

  if (isActive())
    {
      QColor lighterColor (options->color(Options::TitleBar, true).light (150));
      p2.setPen (lighterColor);
      p2.drawLine (r.x(), 2, r.x() + r.width(), 2);
      int h, s, v;
      lighterColor.hsv (&h, &s, &v);
      s /= 2.0;
      s = (s > 255) ? 255 : (int) s;

      QColor satColor(h, s, v, QColor::Hsv);
      p2.setPen (satColor);
      p2.drawLine (r.x(), 1, r.x() + r.width(), 1);
    }


  p2.setPen(Qt::white);
  if (isActive())
    {
      for (int i = 0; i < BtnCount; i ++)
	{
	  if (button[i] == NULL)
	    continue;
	  if (!button[i]->isVisible())
	    {
	      if (button[i]->pos == ButtonRight)
		drawRightDivider = false;
	      // FIXME: Should be LeftButtonLeft if we had it
	      if (button[i]->pos == LeftButtonRight)
		drawLeftDivider = false;
	      continue;
	    }
	  QRect buttonSize = button[i]->geometry ();
	  p2.setPen(Qt::white);
	  p2.drawLine (buttonSize.x() - 1, TOP_GRABBAR_WIDTH,
		       buttonSize.x() - 1, TOP_GRABBAR_WIDTH + titleHeight);
	  if (button[i]->pos == ButtonRight)
	    continue;
	  else if (button[i]->pos == LeftButtonRight)
	    p2.setPen(g.mid().light(120));
	  else
	    p2.setPen(g.dark());
	  p2.drawLine (buttonSize.x() + buttonSize.width(), TOP_GRABBAR_WIDTH - 1,
		       buttonSize.x() + buttonSize.width(), TOP_GRABBAR_WIDTH + titleHeight);
	}
    }


  // *** Top Left Button Area
  if (drawLeftDivider)
    {
      if (isActive ())
	{
	  p2.setPen(options->color(Options::TitleBar, true).dark (150));
	}
      else
	{
	  p2.setPen(g.mid());
	}
      p2.drawLine (r.x() - 1 , y + 1,
		   r.x() - 1, y + titleHeight + TOP_GRABBAR_WIDTH);
    }



  // *** Top Right Button Area
  if (drawRightDivider)
    {
      if (isActive ())
	{
	  p2.setPen(options->color(Options::TitleBar, true).dark (150));
	}
      else
	{
	  p2.setPen(g.mid());
	}
      p2.drawLine (r.x() + r.width(), y + 1,
		   r.x() + r.width(), y + titleHeight + TOP_GRABBAR_WIDTH);
    }



  // Black outer line
  p2.setPen(Qt::black);
  p2.drawRect(0,0,w,h);
  p2.drawArc(x, y, BUTTON_DIAM, BUTTON_DIAM, 90*16, 90*16);
  p2.drawArc(x + w - BUTTON_DIAM , y, BUTTON_DIAM, BUTTON_DIAM, 0*16, 90*16);
  p2.end();

  int sideStart = titleHeight + TOP_GRABBAR_WIDTH + 1;

  // *** Draw the left and right sides
  
  // Fill the left side first
  qDrawShadePanel(&p,
		  // We compensate for the top and bottom parts of the bevel
		  // by drawing 1 pixel below and above the frame part
		  x + 1, y + (sideStart - 1),
		  BORDER_WIDTH - 1, h - (sideStart + 2),
		  g, false, 1, &g.brush(QColorGroup::Background));

  // Right Side
  qDrawShadePanel(&p,
		  x2 - (BORDER_WIDTH - 2), y + (sideStart - 1),
		  BORDER_WIDTH - 2, h - (sideStart + 2),
 		  g, false, 1, &g.brush(QColorGroup::Background));

  p.setPen(g.dark());
  p.drawLine(x2 - (BORDER_WIDTH - 1), sideStart,
	     x2 - (BORDER_WIDTH - 1), h - sideStart);

  // *** Draw the bottom
  qDrawShadePanel(&p,
		  x, y2 - (BORDER_WIDTH - 2),
		  w, (BORDER_WIDTH - 2),
		  g, false, 1, &g.brush(QColorGroup::Background));
  p.setPen(g.dark());
  p.drawLine(x, y2 - (BORDER_WIDTH - 1),
	     x2, y2 - (BORDER_WIDTH - 1));

  // *** Line above the app and below the title bar
  p.setPen(g.dark());
  p.drawLine(x, y + titleHeight + TOP_GRABBAR_WIDTH,
	     x2, y + titleHeight + TOP_GRABBAR_WIDTH);

  bitBlt( this, 0, 0, titleBuffer );

  // *** Draw an outer black frame
  p.setPen(Qt::black);
  p.drawRect(0,0,w,h);

  // *** Put on the bottom corners
  p.drawPixmap(0, h - bottomLeftPix->height(),
	       isActive() ? *abottomLeftPix : *bottomLeftPix);
  p.drawPixmap(w - bottomRightPix->width(), h - bottomRightPix->height(), 
	       isActive() ? *abottomRightPix : *bottomRightPix);
  p.end();
}


void RedHatClient::doShape()
{
  // Obtain widget bounds.
  QRect r(rect());
  int x = 0;
  int y = 0;
  int x2 = width() - 1;
  int y2 = height() - 1;
  int w  = width();
  int h  = height();

  int rad = BUTTON_DIAM / 2;
  int dm = BUTTON_DIAM;

  QBitmap mask(w+1, h+1, true);

  QPainter p(&mask);

  p.fillRect(x, y, w+1, h+1, Qt::color1);

  p.eraseRect(x, y, rad, rad);
  p.eraseRect(w-rad+1, 0, rad, rad);

  p.eraseRect(x, h-BOTTOM_CORNER, BOTTOM_CORNER, BOTTOM_CORNER);
  p.eraseRect(w-BOTTOM_CORNER, h-BOTTOM_CORNER, BOTTOM_CORNER, BOTTOM_CORNER);

  p.setPen(Qt::color1);
  p.setBrush(Qt::color1);

  p.drawPie(x, y, dm, dm, 90*16, 90*16);
  p.drawArc(x, y, dm, dm, 90*16, 90*16);

  p.drawPie(w-dm+1, 0, dm, dm, 0*16, 90*16);
  p.drawArc(w-dm+1, 0, dm, dm, 0*16, 90*16);

  p.drawPixmap(x, h - bottomLeftPix->height(), *bottomLeftPix->mask());
  p.drawPixmap(w-bottomRightPix->width()+1, h - bottomRightPix->height(), 
	       *bottomRightPix->mask());

  p.fillRect(x+BOTTOM_CORNER, h - bottomLeftPix->height(),
	     bottomLeftPix->width()-BOTTOM_CORNER,
	     bottomLeftPix->height()-BOTTOM_CORNER,
	     Qt::color1);

  p.fillRect(w-bottomRightPix->width()+1, h - bottomRightPix->height(), 
	     bottomRightPix->width()-BOTTOM_CORNER,
	     bottomRightPix->height()-BOTTOM_CORNER,
	     Qt::color1);
    

  p.end();
  setMask(mask);
}


void RedHatClient::showEvent(QShowEvent *ev)
{
  calcHiddenButtons();
  doShape();
  Client::showEvent(ev);
}


void RedHatClient::mouseDoubleClickEvent( QMouseEvent * e )
{
  if (titlebar->geometry().contains( e->pos() ) )
    workspace()->performWindowOperation( this, options->operationTitlebarDblClick() );
}


void RedHatClient::maximizeChange(bool m)
{
  if (button[BtnMax]) {
    button[BtnMax]->setBitmap(m ? minmax_bits : maximize_bits);
    button[BtnMax]->setTipText(m ? i18n("Restore") : i18n("Maximize"));
  }
}


void RedHatClient::activeChange(bool)
{
  for(int i=RedHatClient::BtnHelp; i < RedHatClient::BtnCount; i++)
    if(button[i])
      button[i]->repaint(false);
  repaint(false);
}


// The hiding button while shrinking, show button while expanding magic
void RedHatClient::calcHiddenButtons()
{
  // Hide buttons in this order:
  // Sticky, Help, Maximize, Minimize, Close, Menu.
  RedHatButton* btnArray[] = { button[BtnSticky], button[BtnHelp],
			       button[BtnMax], button[BtnIconify], button[BtnClose],
			       button[BtnMenu] };

  int minwidth  = 160; // Start hiding at this width
  int btn_width = 16;
  int current_width = width();
  int count = 0;
  int i;

  // Find out how many buttons we need to hide.
  while (current_width < minwidth)
    {
      current_width += btn_width;
      count++;
    }

  // Bound the number of buttons to hide
  if (count > 6) count = 6;

  // Hide the required buttons...
  for(i = 0; i < count; i++)
    {
      if (btnArray[i] && btnArray[i]->isVisible() )
	btnArray[i]->hide();
    }

  // Show the rest of the buttons...
  for(i = count; i < 6; i++)
    {
      if (btnArray[i] && (!btnArray[i]->isVisible()) )
	btnArray[i]->show();
    }
}


Client::MousePosition RedHatClient::mousePosition( const QPoint& p ) const
{
  MousePosition m = Nowhere;

  // Modify the mouse position if we are using a grab bar.
  if (showGrabBar && (!isTool()) )
    if (p.y() < (height() - 8))
      m = Client::mousePosition(p);
    else
      {
	if (p.x() >= (width() - 20))
	  m = BottomRight;
	else if (p.x() <= 20)
	  m = BottomLeft;
	else
	  m = Bottom;
      }
  else
    m = Client::mousePosition(p);

  return m;
}


// Make sure the menu button follows double click conventions set in kcontrol
void RedHatClient::menuButtonPressed()
{
   QPoint menupoint ( button[BtnMenu]->rect().bottomLeft().x()-1,
                      button[BtnMenu]->rect().bottomLeft().y()+2 );
   workspace()->showWindowMenu( button[BtnMenu]->mapToGlobal( menupoint ), this );
   button[BtnMenu]->setDown(false);
}


// Extended KWin plugin interface
extern "C"
{
  Client *allocate(Workspace *ws, WId w, int)
  {
    return(new RedHatClient(ws, w));
  }

  void init()
  {
    clientHandler = new RedHatHandler();
  }

  void reset()
  {
    clientHandler->reset();
  }

  void deinit()
  {
    delete clientHandler;
  }
}

#include "redhatclient.moc"
// vim: ts=4






#if 0


  // Draw the title text on the pixmap, and with a smaller font
  // for toolwindows than the default.
  QFont fnt = options->font(true);

  if ( isTool() )
    fnt.setPointSize( fnt.pointSize()-2 );  // Shrink font by 2pt

  p2.setFont( fnt );
  // Draw the titlebar stipple if active and available
  if (isActive() && titlePix)
    {
      QFontMetrics fm(fnt);
      int captionWidth = fm.width(caption());
      ///      p2.drawTiledPixmap( captionWidth+5, 0, r.width()-captionWidth-4,
      ///			  titleHeight+1, *titlePix );
    }

  p2.setPen( options->color(Options::Font, isActive()) );
  p2.drawText(2, 1, r.width(), r.height(),
	      AlignLeft | AlignVCenter, caption() );

  bitBlt( this, r.x(), 2, titleBuffer );

  p2.end();

  r = titlebar->geometry();

  // Ensure a shaded window has no unpainted areas

  p.setPen(c2);
  ///  p.drawLine(x+4, y+titleHeight+4, x2-4, y+titleHeight+4);
#endif
