/* -*- Mode: c++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */

#include <iostream>
using std::cerr; using std::endl;
/*
  Bluecurve style.
  Copyright (c) 2002 Red Hat, Inc.
  Authors: Bernhard Rosenkrnzer <bero@redhat.com>
           Preston Brown <pbrown@redhat.com>
           Than Ngo <than@redhat.com>

  Released under the GNU General Public License (GPL) v2.

  Based on the KDE Light style, 2nd revision:
  Copyright (c) 2000-2001 Trolltech AS (info@trolltech.com)

  Permission is hereby granted, free of charge, to any person obtaining a
  copy of this software and associated documentation files (the "Software"),
  to deal in the Software without restriction, including without limitation
  the rights to use, copy, modify, merge, publish, distribute, sublicense,
  and/or sell copies of the Software, and to permit persons to whom the
  Software is furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
  DEALINGS IN THE SOFTWARE.
*/

#include "bluecurve.moc"

#include "qmenubar.h"
#include "qapplication.h"
#include "qpainter.h"
#include "qpalette.h"
#include "qframe.h"
#include "qpushbutton.h"
#include "qdrawutil.h"
#include "qscrollbar.h"
#include "qtabbar.h"
#include "qguardedptr.h"
#include "qlayout.h"
#include "qlineedit.h"
#include "qlistview.h"
#include "qbitmap.h"
#include "qimage.h"
#include "qcombobox.h"
#include "qslider.h"
#include "qstylefactory.h"
#include "qcleanuphandler.h"
#include "qcheckbox.h"
#include "qradiobutton.h"

#define MAX(x, y) ((x) > (y) ? (x) : (y))
#define MIN(x, y) ((x) < (y) ? (x) : (y))

struct BluecurveStylePrivate
{
    BluecurveStylePrivate()
        : hoverWidget(0), hovering(FALSE), sliderActive(FALSE), mousePressed(FALSE),
          scrollbarElement(0), lastElement(0), ref(1)
    { ; }

    QGuardedPtr<QWidget> hoverWidget;
    bool hovering, sliderActive, mousePressed;
    int scrollbarElement, lastElement, ref;
    QPoint mousePos;
};

static BluecurveStylePrivate * singleton = 0;


BluecurveStyle::BluecurveStyle() : QCommonStyle()
{
    if ( !singleton )
        singleton = new BluecurveStylePrivate;
    else
        singleton->ref++;

	basestyle = QStyleFactory::create( "MotifPlus" );
	if ( ! basestyle )
		basestyle = QStyleFactory::create( QStyleFactory::keys().first() );
	if ( ! basestyle )
		qFatal( "BluecurveStyle: couldn't find a base style!" );
}

BluecurveStyle::~BluecurveStyle()
{
    if ( singleton && singleton->ref-- <= 0) {
        delete singleton;
        singleton = 0;
    }
    delete basestyle;
}

void BluecurveStyle::polishPopupMenu( QPopupMenu * )
{
    // empty to satisy pure virtual requirements
}

/* from Qt's motifplus style */
void BluecurveStyle::polish(QWidget *widget)
{
  if (widget->inherits("QButton") ||
      widget->inherits("QComboBox"))
    widget->installEventFilter(this);

  if (widget->inherits("QScrollBar") ||
      widget->inherits("QSlider")) {
    widget->setMouseTracking(TRUE);
    widget->installEventFilter(this);
  }
  QCommonStyle::polish(widget);
}

void BluecurveStyle::unPolish(QWidget *widget)
{
    widget->removeEventFilter(this);
    QCommonStyle::unPolish(widget);
}

bool BluecurveStyle::eventFilter(QObject *object, QEvent *event)
{

    switch(event->type()) {
    case QEvent::MouseButtonPress:
        {
            singleton->mousePressed = TRUE;

            if (! object->inherits("QSlider"))
                break;

            singleton->sliderActive = TRUE;
            break;
        }

    case QEvent::MouseButtonRelease:
        {
            singleton->mousePressed = FALSE;

            if (! object->inherits("QSlider"))
                break;

            singleton->sliderActive = FALSE;
            ((QWidget *) object)->repaint(FALSE);
            break;
        }

    case QEvent::Enter:
        {
            if (! object->isWidgetType())
                break;

            singleton->hoverWidget = (QWidget *) object;
            if (! singleton->hoverWidget->isEnabled()) {
                singleton->hoverWidget = 0;
                break;
            }
            singleton->hoverWidget->repaint(FALSE);
            break;
        }

    case QEvent::Leave:
        {
            if (object != singleton->hoverWidget)
                break;
            QWidget *w = singleton->hoverWidget;
            singleton->hoverWidget = 0;
            w->repaint(FALSE);
            break;
        }

    case QEvent::MouseMove:
        {
            if (! object->isWidgetType() || object != singleton->hoverWidget)
                break;

            if (! object->inherits("QScrollBar") && ! object->inherits("QSlider"
))
                break;

            singleton->mousePos = ((QMouseEvent *) event)->pos();
            if (! singleton->mousePressed) {
                singleton->hovering = TRUE;
                singleton->hoverWidget->repaint(FALSE);
                singleton->hovering = FALSE;
            }

            break;
        }

    default:
        break;
    }

    return QCommonStyle::eventFilter(object, event);
}

static void drawLightBevel(QPainter *p, const QRect &r, const QColorGroup &cg,
			   QStyle::SFlags flags, const QBrush *fill = 0)
{
    QRect br = r;
    bool sunken = (flags & (QStyle::Style_Down | QStyle::Style_On |
			    QStyle::Style_Sunken));

    p->setPen(cg.dark());
    p->drawRect(r);

    if (flags & (QStyle::Style_Down | QStyle::Style_On |
		 QStyle::Style_Sunken | QStyle::Style_Raised)) {
	// button bevel
	p->setPen(sunken ? cg.mid() : cg.light());
	p->drawLine(r.x() + 1, r.y() + 2,
		    r.x() + 1, r.y() + r.height() - 3); // left
	p->drawLine(r.x() + 1, r.y() + 1,
		    r.x() + r.width() - 2, r.y() + 1); // top

	p->setPen(sunken ? cg.light() : cg.mid());
	p->drawLine(r.x() + r.width() - 2, r.y() + 2,
		    r.x() + r.width() - 2, r.y() + r.height() - 3); // right
	p->drawLine(r.x() + 1, r.y() + r.height() - 2,
		    r.x() + r.width() - 2, r.y() + r.height() - 2); // bottom

	br.addCoords(2, 2, -2, -2);
    } else
	br.addCoords(1, 1, -1, -1);

    // fill
    if (fill) p->fillRect(br, *fill);
}

void BluecurveStyle::drawPrimitive(	PrimitiveElement pe,
					QPainter *p,
					const QRect &r,
					const QColorGroup &cg,
					SFlags flags,
					const QStyleOption &data) const
{
    switch (pe) {
    case PE_HeaderSection:
    {
	const QBrush *fill;
	flags = ((flags | Style_Sunken) ^ Style_Sunken) | Style_Raised;
	if (! (flags & Style_Down) && (flags & Style_On))
	    fill = &cg.brush(QColorGroup::Midlight);
	else
	    fill = &cg.brush(QColorGroup::Button);
	drawLightBevel(p, r, cg, flags, fill);
	break;
    }
    case PE_HeaderArrow:
    {
		
	drawPrimitive((flags & Style_Up) ? PE_ArrowUp : PE_ArrowDown, p,
		      r, cg, Style_Enabled);
	break;
    }
    case PE_ButtonCommand:
    case PE_ButtonBevel:
    case PE_ButtonTool:
    {
	const QBrush *fill;
	if (flags & QStyle::Style_Enabled) {
	    if ((flags & (QStyle::Style_Down |
			 QStyle::Style_On |
			 QStyle::Style_Sunken)) ||
		(flags & QStyle::Style_MouseOver))
		fill = &cg.brush(QColorGroup::Midlight);
	    else
		fill = &cg.brush(QColorGroup::Button);
	} else
	    fill = &cg.brush(QColorGroup::Background);
	drawLightBevel(p, r, cg, flags, fill);
	break;
    }

    case PE_ButtonDropDown:
    {
	QBrush thefill;
	bool sunken =
	    (flags & (QStyle::Style_Down | QStyle::Style_On | QStyle::Style_Sunken));

	if (flags & QStyle::Style_Enabled) {
	    if (sunken)
		thefill = cg.brush(QColorGroup::Midlight);
	    else
		thefill = cg.brush(QColorGroup::Button);
	} else
	    thefill = cg.brush(QColorGroup::Background);

	p->setPen(cg.dark());
	p->drawLine(r.topLeft(),	 r.topRight());
	p->drawLine(r.topRight(),	r.bottomRight());
	p->drawLine(r.bottomRight(), r.bottomLeft());

	if (flags & (QStyle::Style_Down | QStyle::Style_On |
		     QStyle::Style_Sunken | QStyle::Style_Raised)) {
	    // button bevel
	    if (sunken)
		p->setPen(cg.mid());
	    else
		p->setPen(cg.light());

	    p->drawLine(r.x(), r.y() + 2,
			r.x(), r.y() + r.height() - 3); // left
	    p->drawLine(r.x(), r.y() + 1,
			r.x() + r.width() - 2, r.y() + 1); // top

	    if (sunken)
		p->setPen(cg.light());
	    else
		p->setPen(cg.mid());

	    p->drawLine(r.x() + r.width() - 2, r.y() + 2,
			r.x() + r.width() - 2, r.y() + r.height() - 3); // right
	    p->drawLine(r.x() + 1, r.y() + r.height() - 2,
			r.x() + r.width() - 2, r.y() + r.height() - 2); // bottom
	}

	p->fillRect(r.x() + 1, r.y() + 2, r.width() - 3, r.height() - 4, thefill);
	break;
    }

    case PE_ButtonDefault:
    {
	p->setPen(cg.shadow());
	p->setBrush(NoBrush);
	p->drawRect(r);
	break;
    }

//     case PE_CheckMark:
//     {
// 	QRect mark(r.x()+(r.width()-6)/2, r.y()+(r.height()-6)/2, 6, 6);
// 	p->setBrush((flags & Style_Active) ? cg.highlightedText() : cg.highlight());
// 	p->setPen(Qt::NoPen);
// 	p->drawEllipse(mark);
// 	break;
//     }

    case PE_Indicator:
	p->setPen(cg.dark());
    if (flags & QStyle::Style_Enabled) {
        if (flags & QStyle::Style_Down)
            p->setBrush(cg.mid());
        else
            p->setBrush(cg.base());
    } else {
        p->setBrush(cg.background());
    }

	p->drawRect(r);
	if(!(flags & Style_Off)) {
	    QPointArray a(7*2);
	    int i, xx, yy;
	    xx = r.x() + 3;
	    yy = r.y() + 5;
	    for(i=0; i<3; i++) {
		a.setPoint(2*i, xx, yy);
		a.setPoint(2*i+1, xx, yy+2);
		xx++; yy++;
	    }
	    yy -= 2;
	    for(i=3; i<7; i++) {
		a.setPoint(2*i, xx, yy);
		a.setPoint(2*i+1, xx, yy+2);
		xx++; yy--;
	    }
        if (flags & QStyle::Style_Enabled)
            p->setPen(cg.highlight());
        else
            p->setPen(cg.mid());
	    p->drawLineSegments(a);
	}
	break;

    case PE_ExclusiveIndicator:
    {
	QRect br = r, // bevel rect
	    cr = r, // contents rect
	    ir = r; // indicator rect
	br.addCoords(1, 1, -1, -1);
	cr.addCoords(2, 2, -2, -2);
	ir.addCoords(3, 3, -3, -3);

	if (flags & Style_MouseOver)
	    p->fillRect(r, cg.brush(QColorGroup::Midlight));
	else
	    p->fillRect(r, cg.brush(QColorGroup::Background));

	p->setPen(flags & Style_Down ? cg.mid() :
		  (flags & Style_Enabled ? cg.base() : cg.background()));
	p->setBrush(flags & Style_Down ? cg.mid() :
		    (flags & Style_Enabled ? cg.base() : cg.background()));
	p->drawEllipse(r);

	p->setPen(cg.dark());
	p->drawArc(r, 0, 16*360);


	if (flags & Style_On) {
	    p->setBrush(cg.highlight());
	    p->drawEllipse(ir);
	}

	break;
    }

    case PE_DockWindowHandle:
    {
	QString title;
	bool drawTitle = FALSE;
	if ( p && p->device()->devType() == QInternal::Widget ) {
	    QWidget *w = (QWidget *) p->device();
	    QWidget *p = w->parentWidget();
	    if (p->inherits("QDockWindow") && ! p->inherits("QToolBar")) {
		drawTitle = TRUE;
		title = p->caption();
	    }
	}

	flags |= Style_Raised;
	if (flags & Style_Horizontal) {
	    if (drawTitle) {
		QPixmap pm(r.height(), r.width());
		QPainter p2(&pm);
		p2.fillRect(0, 0, pm.width(), pm.height(),
			    cg.brush(QColorGroup::Highlight));
		p2.setPen(cg.highlightedText());
		p2.drawText(0, 0, pm.width(), pm.height(), AlignCenter, title);
		p2.end();

		QWMatrix m;
		m.rotate(270.0);
		pm = pm.xForm(m);
		p->drawPixmap(r.x(), r.y(), pm);
	    } else {
		p->fillRect(r, cg.button());
		p->setPen(cg.mid());		
		p->drawLine(r.right() - 1, r.top() + 1,
			    r.right() - 1, r.bottom() - 2);
		p->setPen(cg.light());
		p->drawLine(r.right(), r.top() + 1,
			    r.right(), r.bottom() - 2);

		int yy = r.top() + 3;
		int nLines = (r.height() - 4) / 4;
		
		
		for (int i = 0; i < nLines; yy += 4, i++) {
		    p->setPen(cg.mid().dark());
		    p->drawLine(1, yy + 3,
				1 + 3, yy);
		    p->setPen(cg.light());
		    p->drawLine(2, yy + 3,
				1 + 3, yy + 1);
		}
	    }
	} else {
	    if (drawTitle) {
		p->fillRect(r, cg.brush(QColorGroup::Highlight));
		p->setPen(cg.highlightedText());
		p->drawText(r, AlignCenter, title);
	    } else {
		p->fillRect(r, cg.background());
		p->setPen(cg.mid().dark());
		p->drawLine(r.left() + 2,  r.bottom() - 6,
			    r.right() - 2, r.bottom() - 6);
		p->drawLine(r.left() + 2,  r.bottom() - 3,
			    r.right() - 2, r.bottom() - 3);
		p->setPen(cg.light());
		p->drawLine(r.left() + 2,  r.bottom() - 5,
			    r.right() - 2, r.bottom() - 5);
		p->drawLine(r.left() + 2,  r.bottom() - 2,
			    r.right() - 2, r.bottom() - 2);
	    }
	}
	break;
    }

    case PE_DockWindowSeparator:
    {
	if (r.width() > 20 || r.height() > 20) {
	    if (flags & Style_Horizontal) {
		p->setPen(cg.mid().dark(120));
		p->drawLine(r.left() + 1, r.top() + 6, r.left() + 1, r.bottom() - 6);
		p->setPen(cg.light());
		p->drawLine(r.left() + 2, r.top() + 6, r.left() + 2, r.bottom() - 6);
	    } else {
		p->setPen(cg.mid().dark(120));
		p->drawLine(r.left() + 6, r.top() + 1, r.right() - 6, r.top() + 1);
		p->setPen(cg.light());
		p->drawLine(r.left() + 6, r.top() + 2, r.right() - 6, r.top() + 2);
	    }
	} else
	    QCommonStyle::drawPrimitive(pe, p, r, cg, flags, data);
	break;
    }

    case PE_Splitter:
	if(flags & Style_Horizontal) {
	    int y_mid = r.height()/2;
	    for (int i=0; i< 21; i=i+5) {
		p->setPen(cg.mid().dark());
		p->drawLine(r.x()+2, y_mid-10+i, r.right()-2, y_mid-10+i-3);
		p->setPen(cg.light());
		p->drawLine(r.x()+3, y_mid-10+i, r.right()-1, y_mid-10+i-3);
	    }
	}
	else {
	    int x_mid = r.width()/2;
	    for (int i=0; i< 21; i=i+5) {
		p->setPen(cg.mid().dark());
		p->drawLine(x_mid-10+i+3, r.y()+2, x_mid-10+i, r.bottom()-2);
		p->setPen(cg.light());
		p->drawLine(x_mid-10+i+4, r.y()+2, x_mid-10+i+1, r.bottom()-2);
	    }
	}
		
	if (flags & Style_Horizontal)
	    flags &= ~Style_Horizontal;
	else
	    flags |= Style_Horizontal;
	break;
	// fall through intended

    case PE_DockWindowResizeHandle:
    {
	p->fillRect(r, cg.background());
	if (flags & Style_Horizontal) {
	    p->setPen(cg.highlight().light());
	    p->drawLine(r.left() + 1, r.top() + 1, r.right() - 1, r.top() + 1);
	    p->setPen(cg.highlight());
	    p->drawLine(r.left() + 1, r.top() + 2, r.right() - 1, r.top() + 2);
	    p->setPen(cg.highlight().dark());
	    p->drawLine(r.left() + 1, r.top() + 3, r.right() - 1, r.top() + 3);
	} else {
	    p->setPen(cg.highlight().light());
	    p->drawLine(r.left() + 1, r.top() + 1, r.left() + 1, r.bottom() - 1);
	    p->setPen(cg.highlight());
	    p->drawLine(r.left() + 2, r.top() + 1, r.left() + 2, r.bottom() - 1);
	    p->setPen(cg.highlight().dark());
	    p->drawLine(r.left() + 3, r.top() + 1, r.left() + 3, r.bottom() - 1);
	}
	break;
    }

    case PE_Panel:
    case PE_PanelPopup:
    case PE_PanelLineEdit:
    case PE_PanelTabWidget:
    case PE_WindowFrame:
    {
	int lw = data.isDefault() ?
	    pixelMetric(PM_DefaultFrameWidth) : data.lineWidth();

	if ( ! ( flags & Style_Sunken ) )
	    flags |= Style_Raised;
	if (lw == 2)
	    drawLightBevel(p, r, cg, flags);
	else
	    QCommonStyle::drawPrimitive(pe, p, r, cg, flags, data);
	break;
    }

    case PE_PanelDockWindow:
    {
	int lw = data.isDefault() ?
	    pixelMetric(PM_DockWindowFrameWidth) : data.lineWidth();
	if (lw == 2)
	  if (flags & Style_MouseOver)
	    drawLightBevel(p, r, cg, flags | Style_Raised,
			   &cg.brush(QColorGroup::Midlight));
	  else
	    drawLightBevel(p, r, cg, flags | Style_Raised,
			   &cg.brush(QColorGroup::Button));
	else
	    QCommonStyle::drawPrimitive(pe, p, r, cg, flags, data);
	break;
    }

    case PE_PanelMenuBar:
    {
	int lw = data.isDefault() ?
	    pixelMetric(PM_MenuBarFrameWidth) : data.lineWidth();

	if (lw == 2) {
	    p->fillRect(r, cg.button());
	    p->setPen(cg.light());
	    p->drawLine(r.left(), r.top(), r.right(), r.top());
	    p->setPen(cg.mid());
	    p->drawLine(r.left(), r.bottom(), r.right(), r.bottom());
	}  else
	    QCommonStyle::drawPrimitive(pe, p, r, cg, flags, data);

	
	break;
    }

    case PE_ScrollBarAddLine:
    case PE_ScrollBarSubLine:
    {
	drawLightBevel(p, r, cg, ((flags | Style_Down) ^ Style_Down) |
		       ((flags & Style_Enabled) ? Style_Raised : Style_Default),
		       &cg.brush(QColorGroup::Button));

	if ((pe == PE_ScrollBarAddLine) && (flags & Style_Horizontal)) {
	    pe = PE_ArrowRight;
	} else if(pe == PE_ScrollBarAddLine) {
	    pe = PE_ArrowDown;
	} else if(flags & Style_Horizontal) {
	    pe = PE_ArrowLeft;
	} else {
	    pe = PE_ArrowUp;
	}
	drawPrimitive(pe, p, r, cg, flags);
	break;
    }

    case PE_ScrollBarSubPage:
    case PE_ScrollBarAddPage:
    {
	p->fillRect(r, cg.brush((flags & Style_Down) ?
				 QColorGroup::Midlight :
				 QColorGroup::Mid));
	p->setPen(cg.dark().light());
	if (flags & Style_Horizontal) {
	    p->drawLine(r.left(), r.top(), r.right(), r.top());
	    p->drawLine(r.left(), r.bottom(), r.right(), r.bottom());
	} else {
	    p->drawLine(r.left(), r.top(), r.left(), r.bottom());
	    p->drawLine(r.right(), r.top(), r.right(), r.bottom());
	}
	break;
    }

    case PE_ScrollBarSlider:
    {
	int x1, y1;
	p->setPen(cg.dark());
	drawLightBevel(p, r, cg, ((flags | Style_Down) ^ Style_Down) |
		       ((flags & Style_Enabled) ? Style_Raised : Style_Default),
		       &cg.brush(QColorGroup::Button));

	if (flags & Style_Horizontal && r.width() < 31)
	    break;
	if (!(flags & Style_Horizontal) && r.height() < 31)
	    break;
	p->setPen(cg.mid());
	if (flags & Style_Horizontal) {
	    x1 = (r.left() + r.right()) / 2 - 8;
	    y1 = ((r.top() + r.bottom()) - 6) / 2;
	    p->drawLine(x1 + 5, y1, 
			x1, y1 + 5);
	    p->drawLine(x1 + 5 + 5, y1,
			x1 + 5, y1 + 5);
	    p->drawLine(x1 + 5 + 5*2, y1,
			x1 + 5*2, y1 + 5);
	} else {
	    x1 = ((r.left() + r.right()) - 6) / 2;
	    y1 = (r.top() + r.bottom()) / 2 - 8;
	    p->drawLine(x1 + 5, y1,
			x1, y1 + 5);
	    p->drawLine(x1 + 5, y1 + 5,
			x1, y1 + 5 + 5);
	    p->drawLine(x1 + 5, y1 + 5*2,
			x1, y1 + 5 + 5*2);
	}

	p->setPen(cg.light());
	if (flags & Style_Horizontal) {
	    x1 = (r.left() + r.right()) / 2 - 8;
	    y1 = ((r.top() + r.bottom()) - 6) / 2;
	    p->drawLine(x1 + 5, y1+1, 
			x1 + 1, y1 + 5);
	    p->drawLine(x1 + 5 + 5, y1 + 1,
			x1 + 1 + 5, y1 + 5);
	    p->drawLine(x1 + 5 + 5*2, y1 + 1,
			x1 + 1 + 5*2, y1 + 5);
	} else {
	    x1 = ((r.left() + r.right()) - 6) / 2;
	    y1 = (r.top() + r.bottom()) / 2 - 8;
	    p->drawLine(x1 + 5, y1 + 1,
			x1 + 1, y1 + 5);
	    p->drawLine(x1 + 5, y1 + 1 + 5,
			x1 + 1, y1 + 5 + 5);
	    p->drawLine(x1 + 5, y1 + 1 + 5*2,
			x1 + 1, y1 + 5 + 5*2);
	}

	break;
    }

    case PE_FocusRect:
    {
	p->drawWinFocusRect(r);
	break;
    }

    case PE_ProgressBarChunk:
    {
	//QRect grad(r.left()+2, r.top()+2, r.width()-3, r.height()-3);
	drawGradient(p, r, cg.highlight(), false, 0, 0, 80);	
	break;
    }

    case PE_ArrowUp:
    case PE_ArrowDown:
    case PE_ArrowRight:
    case PE_ArrowLeft:
    {
	QPointArray a;

	switch ( pe ) {
	case PE_ArrowUp:
	    a.setPoints(11,   3,1,  0,-2,  -3,1,  -2,0,  -2,2,  0,-1,  2,1,  2,2,  0,0,  -1,1,  1,1);
	    break;

	case PE_ArrowDown:
	    a.setPoints(11,   3,-1,  0,2,  -3,-1,  -2,0,  -2,-2,  0,1,  2,-1,  2,-2,  0,0,  -1,-1,  1,-1);
	    break;

	case PE_ArrowRight:
	    a.setPoints(13,  0,-3,  -1,-2,  1,-2,  2,-1,  0,-1,  1,0,  3,0,  2,1,  0,1,  -1,2,  1,2,  0,3,  0,0);
	    break;

	case PE_ArrowLeft:
	    a.setPoints(13,  0,-3,  1,-2,  -1,-2,  -2,-1,  0,-1,  -1,0,  -3,0,  -2,1,  0,1,  1,2,  -1,2,  0,3,  0,0);
	    break;

	default:
	    break;
	}

	if (a.isNull())
	    return;

	p->save();
	a.translate( (r.x() + r.width() / 2), 
		     (r.y() + r.height() / 2));
	if ( flags & Style_Enabled ) {
	    p->setPen( flags & (Style_Active | Style_MouseOver) ? cg.highlightedText() : cg.buttonText() );
	} else {
	    p->setPen( cg.mid() );
	}
	p->drawPolyline(a);
	p->restore();

	break;
    }

    case PE_SpinWidgetUp:
    case PE_SpinWidgetDown:
    {
	QCommonStyle::drawPrimitive(PE_ButtonDefault, p, r, cg, flags, data);
	p->setPen(cg.shadow());
	p->setBrush(QBrush::NoBrush);
	QPointArray a;
	if(pe==PE_SpinWidgetUp) {
	    a.setPoints(6,  -3,0,  -1,-2,  1,0,  1,1,  -1,-1,  -3,1);
	} else {
	    a.setPoints(6,  -3,-1,  -1,1,  1,-1,  1,-2,  -1,0,  -3,-2);
	}
	a.translate( r.x() + r.width() / 2, r.y() + r.height() / 2 );
	p->drawPolyline(a);
	break;
    }

    default:
	QCommonStyle::drawPrimitive(pe, p, r, cg, flags, data);
	break;
    }
}

void BluecurveStyle::drawControl(ControlElement control,
				  QPainter *p,
				  const QWidget *widget,
				  const QRect &r,
				  const QColorGroup &cg,
				  SFlags flags,
				  const QStyleOption &data) const
{
    if (widget == singleton->hoverWidget) {
        flags |= Style_MouseOver;
    }

    switch (control) {
    case CE_TabBarTab:
    {
	QRect tr(r);
	QRect fr(r);

	tr.addCoords(0, 0,  0, -1);
	fr.addCoords(2, 2, -2, -2);

	if (! (flags & Style_Selected)) {
	    tr.addCoords(0, 1, 0, 0);
	    fr.addCoords(0, 1, 0, 0);

	    p->setPen(cg.dark());
	    p->drawRect(tr);

	    if (tr.left() == 0)
		p->drawPoint(tr.left(), tr.bottom() + 1);

	    p->setPen(cg.light());
	    p->drawLine(tr.left() + 1, tr.bottom() - 1,
			tr.left() + 1, tr.top() + 2);
	    p->drawLine(tr.left() + 1, tr.top() + 1,
			tr.right() - 1, tr.top() + 1);
	    if (tr.left() == 0)
		p->drawLine(tr.left() + 1, tr.bottom() + 1,
			    tr.right(), tr.bottom() + 1);
	    else
		p->drawLine(tr.left(), tr.bottom() + 1,
			    tr.right(), tr.bottom() + 1);

	    p->setPen(cg.mid());
	    p->drawLine(tr.right() - 1, tr.top() + 2,
			tr.right() - 1, tr.bottom() - 1);
	} else {
	    p->setPen(cg.dark());
	    if (tr.left() == 0)
		p->drawLine(tr.left(), tr.bottom() + 1,
			    tr.left(), tr.top() + 1);
	    else
		p->drawLine(tr.left(), tr.bottom(),
			    tr.left(), tr.top() + 1);
	    p->drawLine(tr.left(), tr.top(),
			tr.right(), tr.top());
	    p->drawLine(tr.right(), tr.top() + 1,
			tr.right(), tr.bottom());

	    p->setPen(cg.light());
	    if (tr.left() == 0)
		p->drawLine(tr.left() + 1, tr.bottom() + 2,
			    tr.left() + 1, tr.top() + 2);
	    else {
		p->drawLine(tr.left() + 1, tr.bottom(),
			    tr.left() + 1, tr.top() + 2);
		p->drawPoint(tr.left(), tr.bottom() + 1);
	    }
	    p->drawLine(tr.left() + 1, tr.top() + 1,
			tr.right() - 1, tr.top() + 1);
	    p->drawPoint(tr.right(), tr.bottom() + 1);

	    p->setPen(cg.mid());
	    p->drawLine(tr.right() - 1, tr.top() + 2,
			tr.right() - 1, tr.bottom());
	}

	p->fillRect(fr, ((flags & Style_Selected) ?
			 cg.background() : cg.mid()));
	break;
    }

    case CE_PopupMenuItem:
    {
	if (! widget || data.isDefault())
	    break;

	const QPopupMenu *popupmenu = (const QPopupMenu *) widget;
	QMenuItem *mi = data.menuItem();
	int tab = data.tabWidth();
	int maxpmw = data.maxIconWidth();

	if ( mi && mi->isSeparator() ) {
	    // draw separator
	    p->fillRect(r, cg.brush(QColorGroup::Button));
	    p->setPen(cg.mid().dark(120));
	    p->drawLine(r.left(), r.top() + 1,
			r.right(), r.top() + 1);
	    p->setPen(cg.light());
	    p->drawLine(r.left(),  r.top() + 2,
			r.right(), r.top() + 2);
	    break;
	}

	if((flags & Style_Active) && (flags & Style_Enabled)) {
	    drawGradientBox(p, r, cg.highlight());
	} else
	    p->fillRect(r, cg.brush(QColorGroup::Button));

	if ( !mi )
	    break;

	maxpmw = QMAX(maxpmw, 16);

	QRect cr, ir, tr, sr;
	// check column
	cr.setRect(r.left(), r.top(), maxpmw, r.height());
	// submenu indicator column
	sr.setCoords(r.right() - maxpmw, r.top(), r.right(), r.bottom());
	// tab/accelerator column
	tr.setCoords(sr.left() - tab - 4, r.top(), sr.left(), r.bottom());
	// item column
	ir.setCoords(cr.right() + 4, r.top(), tr.right() - 4, r.bottom());

	bool reverse = QApplication::reverseLayout();
	if ( reverse ) {
	    cr = visualRect( cr, r );
	    sr = visualRect( sr, r );
	    tr = visualRect( tr, r );
	    ir = visualRect( ir, r );
	}

	if (mi->iconSet()) {
	    QIconSet::Mode mode =
		(flags & Style_Enabled) ? QIconSet::Normal : QIconSet::Disabled;
	    if ((flags & Style_Active) && (flags & Style_Enabled))
		mode = QIconSet::Active;
	    QPixmap pixmap;
	    if (popupmenu->isCheckable() && mi->isChecked())
		pixmap=mi->iconSet()->pixmap( QIconSet::Small, mode, QIconSet::On );
	    else
		pixmap=mi->iconSet()->pixmap( QIconSet::Small, mode );
	    QRect pmr(QPoint(0, 0), pixmap.size());
	    pmr.moveCenter(cr.center());
	    p->setPen(cg.text());
	    p->drawPixmap(pmr.topLeft(), pixmap);
	} else if (popupmenu->isCheckable() && mi->isChecked())
	    drawPrimitive(PE_CheckMark, p, cr, cg,
			  (flags & (Style_Enabled|Style_Active)) | Style_On);

	QColor textcolor;
	QColor embosscolor;
	if (flags & Style_Active) {
	    if (! (flags & Style_Enabled)) {
            textcolor = cg.text();
            embosscolor = cg.light();
        } else {
            textcolor = cg.highlightedText();
            embosscolor = cg.midlight().light();
        }
	} else if (! (flags & Style_Enabled)) {
	    textcolor = cg.text();
	    embosscolor = cg.light();
	} else
	    textcolor = embosscolor = cg.buttonText();
	p->setPen(textcolor);

	if (mi->custom()) {
	    p->save();
	    if (! (flags & Style_Enabled)) {
		p->setPen(cg.light());
		mi->custom()->paint(p, cg, (flags & Style_Enabled) ? (flags & Style_Active) : 0,
				    flags & Style_Enabled,
				    ir.x() + 1, ir.y() + 1,
				    ir.width() - 1, ir.height() - 1);
		p->setPen(textcolor);
	    }
	    mi->custom()->paint(p, cg, (flags & Style_Enabled) ? (flags & Style_Active) : 0,
				flags & Style_Enabled,
				ir.x(), ir.y(),
				ir.width(), ir.height());
	    p->restore();
	}

	QString text = mi->text();
	if (! text.isNull()) {
	    int t = text.find('\t');

	    // draw accelerator/tab-text
	    if (t >= 0) {
		int alignFlag = AlignVCenter | ShowPrefix | DontClip | SingleLine;
		alignFlag |= ( reverse ? AlignLeft : AlignRight );
		if (! (flags & Style_Enabled)) {
		    p->setPen(embosscolor);
		    tr.moveBy(1, 1);
		    p->drawText(tr, alignFlag, text.mid(t + 1));
		    tr.moveBy(-1, -1);
		    p->setPen(textcolor);
		}

		p->drawText(tr, alignFlag, text.mid(t + 1));
	    }

	    int alignFlag = AlignVCenter | ShowPrefix | DontClip | SingleLine;
	    alignFlag |= ( reverse ? AlignRight : AlignLeft );

	    if (! (flags & Style_Enabled)) {
		p->setPen(embosscolor);
		ir.moveBy(1, 1);
		p->drawText(ir, alignFlag, text, t);
		ir.moveBy(-1, -1);
		p->setPen(textcolor);
	    }

	    p->drawText(ir, alignFlag, text, t);
	} else if (mi->pixmap()) {
	    QPixmap pixmap = *mi->pixmap();
	    if (pixmap.depth() == 1)
		p->setBackgroundMode(OpaqueMode);
	    p->drawPixmap(ir.x(), (ir.height() - pixmap.height()) / 2, pixmap);
	    if (pixmap.depth() == 1)
		p->setBackgroundMode(TransparentMode);
	}

	if (mi->popup())
	    drawPrimitive( (reverse ? PE_ArrowLeft : PE_ArrowRight), p, sr, cg, flags);
	break;
    }

    case CE_MenuBarItem:
    {
	if ((flags & Style_Enabled) && (flags & Style_Active) &&
	    (flags & Style_Down))
	    drawGradientBox(p, r, cg.highlight());
	else
	    p->fillRect(r, cg.brush(QColorGroup::Button));

	if (data.isDefault())
	    break;

	QMenuItem *mi = data.menuItem();
	if (flags & Style_Active && (flags & Style_Down))
	    drawItem(p, r, AlignCenter | ShowPrefix | DontClip | SingleLine, cg, flags & Style_Enabled, mi->pixmap(), mi->text(), -1, &cg.highlightedText());
	else
	    drawItem(p, r, AlignCenter | ShowPrefix | DontClip | SingleLine, cg, flags & Style_Enabled, mi->pixmap(), mi->text(), -1, &cg.buttonText());
	break;
    }

    case CE_ProgressBarGroove:
	p->setBrush(cg.brush(QColorGroup::Mid));
	p->setPen(cg.dark());
	p->drawRect(r);
	break;

    case CE_CheckBox:
	drawPrimitive(PE_Indicator, p, r, cg, flags, data);
	break;
    
    case CE_CheckBoxLabel:
      {
          const QCheckBox *checkbox = (const QCheckBox *) widget;
          
          if (flags & Style_MouseOver) {
              QRegion r(checkbox->rect());
              r -= visualRect(subRect(SR_CheckBoxIndicator, widget), widget);
              p->setClipRegion(r);
              p->fillRect(checkbox->rect(), cg.brush(QColorGroup::Midlight));
              p->setClipping(FALSE);
          }
          int alignment = QApplication::reverseLayout() ? AlignRight : AlignLeft;
          drawItem(p, r, alignment | AlignVCenter | ShowPrefix, cg,
                   flags & Style_Enabled, checkbox->pixmap(), checkbox->text());
          
          if (checkbox->hasFocus()) {
              QRect fr = visualRect(subRect(SR_CheckBoxFocusRect, widget), widget);
              drawPrimitive(PE_FocusRect, p, fr, cg, flags);
          }
      }
      break;

    case CE_RadioButtonLabel:
      {
	const QRadioButton *radiobutton = (const QRadioButton *) widget;
	
	if (flags & Style_MouseOver) {
	  QRegion r(radiobutton->rect());
	  r -= visualRect(subRect(SR_RadioButtonIndicator, widget), widget);
	  p->setClipRegion(r);
	  p->fillRect(radiobutton->rect(), cg.brush(QColorGroup::Midlight));
	  p->setClipping(FALSE);
	}
	
	int alignment = QApplication::reverseLayout() ? AlignRight : AlignLeft;
	drawItem(p, r, alignment | AlignVCenter | ShowPrefix, cg,
		 flags & Style_Enabled, radiobutton->pixmap(), radiobutton->text());
	
	if (radiobutton->hasFocus()) {
	  QRect fr = visualRect(subRect(SR_RadioButtonFocusRect, widget), widget);
	  drawPrimitive(PE_FocusRect, p, fr, cg, flags);
	}
      }
      break;
    default:
	QCommonStyle::drawControl(control, p, widget, r, cg, flags, data);
	break;
   }
}

void BluecurveStyle::drawControlMask(ControlElement control,
				      QPainter *p,
				      const QWidget *widget,
				      const QRect &r,
				      const QStyleOption &data) const
{
    switch (control) {
    case CE_PushButton:
	p->fillRect(r, color1);
	break;

    default:
	QCommonStyle::drawControlMask(control, p, widget, r, data);
	break;
    }
}

QRect BluecurveStyle::subRect(SubRect subrect, const QWidget *widget) const
{
    QRect rect, wrect(widget->rect());

    switch (subrect) {
    case SR_PushButtonFocusRect:
    {
	const QPushButton *button = (const QPushButton *) widget;
	int dbw1 = 0, dbw2 = 0;
	if (button->isDefault() || button->autoDefault()) {
	    dbw1 = pixelMetric(PM_ButtonDefaultIndicator, widget);
	    dbw2 = dbw1 * 2;
	}

	rect.setRect(wrect.x()	  + 3 + dbw1,
		     wrect.y()	  + 3 + dbw1,
		     wrect.width()  - 6 - dbw2,
		     wrect.height() - 6 - dbw2);
	break;
    }

    case SR_CheckBoxIndicator:
        {
            int h = pixelMetric( PM_IndicatorHeight );
            rect.setRect(( widget->rect().height() - h ) / 2,
                         ( widget->rect().height() - h ) / 2,
                         pixelMetric( PM_IndicatorWidth ), h );
            break;
        }

    case SR_RadioButtonIndicator:
        {
            int h = pixelMetric( PM_ExclusiveIndicatorHeight );
            rect.setRect( ( widget->rect().height() - h ) / 2,
                          ( widget->rect().height() - h ) / 2,
                          pixelMetric( PM_ExclusiveIndicatorWidth ), h );
            break;
        }

    default:
	rect = QCommonStyle::subRect(subrect, widget);
    }

    return rect;
}

void BluecurveStyle::drawComplexControl(ComplexControl control,
					 QPainter* p,
					 const QWidget* widget,
					 const QRect& r,
					 const QColorGroup& cg,
					 SFlags flags,
					 SCFlags controls,
					 SCFlags active,
					 const QStyleOption &data) const
{
    if (widget == singleton->hoverWidget)
        flags |= Style_MouseOver;

    switch (control) {
    case CC_ListView:
    {
	if ( controls & SC_ListView )
	    QCommonStyle::drawComplexControl( control, p, widget, r, cg, flags, controls, active, data );
	if ( controls & ( SC_ListViewBranch | SC_ListViewExpand ) ) {
	    if (data.isDefault())
		break;

	    QListViewItem *item = data.listViewItem(),
		*child = item->firstChild();

	    int y = r.y();
	    int c;
	    int dotoffset = 0;
	    QPointArray dotlines;
	    if ( active == SC_All && controls == SC_ListViewExpand ) {
		c = 2;
		dotlines.resize(2);
		dotlines[0] = QPoint( r.right(), r.top() );
		dotlines[1] = QPoint( r.right(), r.bottom() );
	    } else {
		int linetop = 0, linebot = 0;
		// each branch needs at most two lines, ie. four end points
		dotoffset = (item->itemPos() + item->height() - y) %2;
		dotlines.resize( item->childCount() * 4 );
		c = 0;

		// skip the stuff above the exposed rectangle
		while ( child && y + child->height() <= 0 ) {
		    y += child->totalHeight();
		    child = child->nextSibling();
		}

		int bx = r.width() / 2;

		// paint stuff in the magical area
		QListView* v = item->listView();
		int lh;
		if ( !item->multiLinesEnabled() )
		    lh = child ? child->height() : 0;
		else
		    lh = p->fontMetrics().height() + 2 * v->itemMargin();
		lh = QMAX( lh, QApplication::globalStrut().height() );
		if ( lh % 2 > 0 )
		    lh++;
		while ( child && y < r.height() ) {
		    linebot = y + lh/2;
		    if ( (child->isExpandable() || child->childCount()) &&
			 (child->height() > 0) ) {
			// needs a box
			p->setPen( cg.mid() );
			p->drawRect( bx-4, linebot-4, 9, 9 );
			// plus or minus
			p->setPen( cg.text() );
			p->drawLine( bx - 2, linebot, bx + 2, linebot );
			if ( !child->isOpen() )
			    p->drawLine( bx, linebot - 2, bx, linebot + 2 );
			// dotlinery
			p->setPen( cg.mid() );
			dotlines[c++] = QPoint( bx, linetop );
			dotlines[c++] = QPoint( bx, linebot - 4 );
			dotlines[c++] = QPoint( bx + 5, linebot );
			dotlines[c++] = QPoint( r.width(), linebot );
			linetop = linebot + 5;
		    } else {
			// just dotlinery
			dotlines[c++] = QPoint( bx+1, linebot );
			dotlines[c++] = QPoint( r.width(), linebot );
		    }

		    y += child->totalHeight();
		    child = child->nextSibling();
		}

		if ( child ) // there's a child, so move linebot to edge of rectangle
		    linebot = r.height();

		if ( linetop < linebot ) {
		    dotlines[c++] = QPoint( bx, linetop );
		    dotlines[c++] = QPoint( bx, linebot );
		}
	    }
	    p->setPen( cg.text() );

	    static QBitmap *verticalLine = 0, *horizontalLine = 0;
	    static QCleanupHandler<QBitmap> qlv_cleanup_bitmap;
	    if ( !verticalLine ) {
		// make 128*1 and 1*128 bitmaps that can be used for
		// drawing the right sort of lines.
		verticalLine = new QBitmap( 1, 129, TRUE );
		horizontalLine = new QBitmap( 128, 1, TRUE );
		QPointArray a( 64 );
		QPainter p;
		p.begin( verticalLine );
		int i;
		for( i=0; i<64; i++ )
		    a.setPoint( i, 0, i*2+1 );
		p.setPen( color1 );
		p.drawPoints( a );
		p.end();
		QApplication::flushX();
		verticalLine->setMask( *verticalLine );
		p.begin( horizontalLine );
		for( i=0; i<64; i++ )
		    a.setPoint( i, i*2+1, 0 );
		p.setPen( color1 );
		p.drawPoints( a );
		p.end();
		QApplication::flushX();
		horizontalLine->setMask( *horizontalLine );
		qlv_cleanup_bitmap.add( &verticalLine );
		qlv_cleanup_bitmap.add( &horizontalLine );
	    }

	    int line; // index into dotlines
	    for( line = 0; line < c; line += 2 ) {
		// assumptions here: lines are horizontal or vertical.
		// lines always start with the numerically lowest
		// coordinate.

		// point ... relevant coordinate of current point
		// end ..... same coordinate of the end of the current line
		// other ... the other coordinate of the current point/line
		if ( dotlines[line].y() == dotlines[line+1].y() ) {
		    int end = dotlines[line+1].x();
		    int point = dotlines[line].x();
		    int other = dotlines[line].y();
		    while( point < end ) {
			int i = 128;
			if ( i+point > end )
			    i = end-point;
			p->drawPixmap( point, other, *horizontalLine,
				       0, 0, i, 1 );
			point += i;
		    }
		} else {
		    int end = dotlines[line+1].y();
		    int point = dotlines[line].y();
		    int other = dotlines[line].x();
		    int pixmapoffset = ((point & 1) != dotoffset ) ? 1 : 0;
		    while( point < end ) {
			int i = 128;
			if ( i+point > end )
			    i = end-point;
			p->drawPixmap( other, point, *verticalLine,
				       0, pixmapoffset, 1, i );
			point += i;
		    }
		}
	    }
	}
	break;
    }
    case CC_ComboBox:
    {
	const QComboBox *combobox = (const QComboBox *) widget;
	QRect frame, arrow, field;

	frame=QStyle::visualRect(querySubControlMetrics(CC_ComboBox, widget, SC_ComboBoxFrame, data), widget);
	arrow=QStyle::visualRect(querySubControlMetrics(CC_ComboBox, widget, SC_ComboBoxArrow, data), widget);
	field=QStyle::visualRect(querySubControlMetrics(CC_ComboBox, widget, SC_ComboBoxEditField, data), widget);

	if ((controls & SC_ComboBoxFrame) && frame.isValid()) {
        if (flags & QStyle::Style_MouseOver)
            drawLightBevel(p, frame, cg, flags | Style_Raised, &cg.brush(QColorGroup::Midlight));
        else
            drawLightBevel(p, frame, cg, flags | Style_Raised, &cg.brush(QColorGroup::Button));
	}

	if ((controls & SC_ComboBoxArrow) && arrow.isValid()) {
	    if (active == SC_ComboBoxArrow)
            p->fillRect(arrow, cg.brush(QColorGroup::Dark));
	    //arrow.addCoords(4, 2, -2, -6);
	    drawPrimitive(PE_ArrowDown, p, arrow, cg, flags & ~Style_MouseOver);
	    p->setPen(cg.mid());
	    p->drawLine((arrow.left()+arrow.right())/2-1, 
                    (arrow.top()+arrow.bottom())/2+5, 
                    (arrow.left()+arrow.right())/2+3,
                    (arrow.top()+arrow.bottom())/2+5);
	    p->drawLine((arrow.left()+arrow.right())/2-1,
                    (arrow.top()+arrow.bottom())/2+6,
                    (arrow.left()+arrow.right())/2+3,
                    (arrow.top()+arrow.bottom())/2+6);
	}

	if ((controls & SC_ComboBoxEditField) && field.isValid()) {
	    p->setPen(cg.mid());
	    if (combobox->editable()) {
            field.addCoords(-1, -1, 1, 1);
            p->drawRect(field);
	    } else
            p->drawLine(field.right() + 1, field.top(),
                        field.right() + 1, field.bottom());

	    if (flags & Style_HasFocus) {
            if (! combobox->editable()) {
                QRect fr=QStyle::visualRect(subRect(SR_ComboBoxFocusRect, widget), widget );
                drawPrimitive(PE_FocusRect, p, fr, cg, flags | Style_FocusAtBorder, QStyleOption(cg.highlight()));
            }

	    }
        if (flags & Style_Enabled)
            p->setPen(cg.buttonText());
        else
            p->setPen(cg.mid());
	}

	break;
    }

    case CC_SpinWidget:
    {
	const QSpinWidget *spinwidget = (const QSpinWidget *) widget;
	QRect frame, up, down;

	frame = querySubControlMetrics(CC_SpinWidget, widget, SC_SpinWidgetFrame, data);
	up = spinwidget->upRect();
	down = spinwidget->downRect();

	if ((controls & SC_SpinWidgetFrame) && frame.isValid())
	    drawLightBevel(p, frame, cg, flags | Style_Sunken, &cg.brush(QColorGroup::Base));

	if ((controls & SC_SpinWidgetUp) && up.isValid()) {
	    PrimitiveElement pe = PE_SpinWidgetUp;
	    if (spinwidget->buttonSymbols() == QSpinWidget::PlusMinus)
		pe = PE_SpinWidgetPlus;

	    p->setPen(cg.dark());
	    p->drawLine(up.topLeft(), up.bottomLeft());

	    up.addCoords(1, 0, 0, 0);
	    p->fillRect(up, cg.brush(QColorGroup::Button));
	    if (active == SC_SpinWidgetUp)
		p->setPen(cg.mid());
	    else
		p->setPen(cg.light());
	    p->drawLine(up.left(), up.top(), up.right(), up.top());
	    p->drawLine(up.left(), up.top(), up.left(), up.bottom());
	    if (active == SC_SpinWidgetUp)
		p->setPen(cg.light());
	    else
		p->setPen(cg.mid());
	    p->drawLine(up.right(), up.top()+1, up.right(), up.bottom());
	    p->drawLine(up.left()+1, up.bottom(), up.right(), up.bottom());

	    up.addCoords(1, 0, 0, 0);
	    drawPrimitive(pe, p, up, cg, flags | ((active == SC_SpinWidgetUp) ? Style_On | Style_Sunken : Style_Raised));
	}

	if ((controls & SC_SpinWidgetDown) && down.isValid()) {
	    PrimitiveElement pe = PE_SpinWidgetDown;
	    if (spinwidget->buttonSymbols() == QSpinWidget::PlusMinus)
		pe = PE_SpinWidgetMinus;

	    p->setPen(cg.dark());
	    p->drawLine(down.topLeft(), down.bottomLeft());

	    down.addCoords(1, 0, 0, 0);
	    p->fillRect(down, cg.brush(QColorGroup::Button));
	    if (active == SC_SpinWidgetDown)
		p->setPen(cg.mid());
	    else
		p->setPen(cg.light());
	    p->drawLine(down.left(), down.top(), down.right(), down.top());
	    p->drawLine(down.left(), down.top(), down.left(), down.bottom());
	    if (active == SC_SpinWidgetDown)
		p->setPen(cg.light());
	    else
		p->setPen(cg.mid());
	    p->drawLine(down.right(), down.top()+1,
			down.right(), down.bottom());
	    p->drawLine(down.left()+1, down.bottom(),
			down.right(), down.bottom());

	    down.addCoords(1, 0, 0, 0);
	    drawPrimitive(pe, p, down, cg, flags |
			  ((active == SC_SpinWidgetDown) ?
			   Style_On | Style_Sunken : Style_Raised));
	}

	break;
    }

    case CC_ScrollBar:
    {
	const QScrollBar *scrollbar = (const QScrollBar *) widget;
	QRect addline, subline, addpage, subpage, slider, first, last;
	bool maxedOut = (scrollbar->minValue() == scrollbar->maxValue());

	subline = querySubControlMetrics(control, widget, SC_ScrollBarSubLine, data);
	addline = querySubControlMetrics(control, widget, SC_ScrollBarAddLine, data);
	subpage = querySubControlMetrics(control, widget, SC_ScrollBarSubPage, data);
	addpage = querySubControlMetrics(control, widget, SC_ScrollBarAddPage, data);
	slider  = querySubControlMetrics(control, widget, SC_ScrollBarSlider,  data);
	first   = querySubControlMetrics(control, widget, SC_ScrollBarFirst,   data);
	last	= querySubControlMetrics(control, widget, SC_ScrollBarLast,	data);

	if ((controls & SC_ScrollBarSubLine) && subline.isValid()) {
	    drawPrimitive(PE_ScrollBarSubLine, p, subline, cg, Style_Enabled | ((active == SC_ScrollBarSubLine) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));

	}
	if ((controls & SC_ScrollBarAddLine) && addline.isValid())
	    drawPrimitive(PE_ScrollBarAddLine, p, addline, cg, Style_Enabled | ((active == SC_ScrollBarAddLine) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));
	if ((controls & SC_ScrollBarSubPage) && subpage.isValid())
	    drawPrimitive(PE_ScrollBarSubPage, p, subpage, cg, Style_Enabled | ((active == SC_ScrollBarSubPage) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));
	if ((controls & SC_ScrollBarAddPage) && addpage.isValid())
	    drawPrimitive(PE_ScrollBarAddPage, p, addpage, cg, ((maxedOut) ? Style_Default : Style_Enabled) | ((active == SC_ScrollBarAddPage) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));
	if ((controls & SC_ScrollBarFirst) && first.isValid())
	    drawPrimitive(PE_ScrollBarFirst, p, first, cg, Style_Enabled | ((active == SC_ScrollBarFirst) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));
	if ((controls & SC_ScrollBarLast) && last.isValid())
	    drawPrimitive(PE_ScrollBarLast, p, last, cg, Style_Enabled | ((active == SC_ScrollBarLast) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));
	if ((controls & SC_ScrollBarSlider) && slider.isValid()) {
	    drawPrimitive(PE_ScrollBarSlider, p, slider, cg, Style_Enabled | ((active == SC_ScrollBarSlider) ? Style_Down : Style_Default) | ((scrollbar->orientation() == Qt::Horizontal) ? Style_Horizontal : 0));

	    // ### perhaps this should not be able to accept focus if maxedOut?
	    if (scrollbar->hasFocus()) {
		QRect fr(slider.x() + 2, slider.y() + 2, slider.width() - 5, slider.height() - 5);
		drawPrimitive(PE_FocusRect, p, fr, cg, Style_Default);
	    }
	}

	break;
    }

    case CC_Slider:
    {
	const QSlider *slider = (const QSlider *) widget;
	QRect groove = querySubControlMetrics(CC_Slider, widget, SC_SliderGroove, data),
	    handle = querySubControlMetrics(CC_Slider, widget, SC_SliderHandle, data);

	if ((controls & SC_SliderGroove) && groove.isValid()) {
	    if (flags & Style_HasFocus)
		drawPrimitive( PE_FocusRect, p, groove, cg );

	    if (slider->orientation() == Qt::Horizontal) {
		int dh = (groove.height() - 5) / 2;
		groove.addCoords(0, dh, 0, -dh);
		handle.addCoords(0, 1, 0, -1);
	    } else {
		int dw = (groove.width() - 5) / 2;
		groove.addCoords(dw, 0, -dw, 0);
		handle.addCoords(1, 0, -1, 0);
	    }

	    p->setPen(cg.dark());
	    p->setBrush(cg.mid());
	    p->drawRect(groove);
	    p->setPen(cg.dark().light());
	    p->drawLine(groove.left()+1, groove.top()+1,
			groove.left()+1, groove.bottom()-1);
	    p->drawLine(groove.left()+1, groove.top()+1,
			groove.right()-1, groove.top()+1);
	}

	if ((controls & SC_SliderHandle) && handle.isValid()) {
	    p->setPen(cg.dark());
	    p->setBrush(cg.button());
	    p->drawRoundRect(handle, 35, 35);
	    p->setBrush(Qt::NoBrush);
	    QRect effect=handle;
	    effect.addCoords(1, 1, 0, 0);
	    p->setPen(cg.base());
	    p->drawRoundRect(effect, 35, 35);
	    p->setPen(cg.dark());
	    p->drawRoundRect(handle, 35, 35);


	    int x1, y1;
	    p->setPen(cg.mid());
	    if (slider->orientation() == Qt::Horizontal) {
		x1 = (handle.left() + handle.right()) / 2 - 8;
		y1 = ((handle.top() + handle.bottom()) - 5) / 2;
		p->drawLine(x1 + 4, y1 + 1, 
			    x1 + 1, y1 + 4);
		p->drawLine(x1 + 5 + 5, y1,
			    x1 + 5, y1 + 5);
		p->drawLine(x1 + 4 + 5*2, y1 + 1,
			    x1 + 1 + 5*2, y1 + 4);
	    } else {
		x1 = ((handle.left() + handle.right()) - 5) / 2;
		y1 = (handle.top() + handle.bottom()) / 2 - 8;
		p->drawLine(x1 + 4, y1 + 1,
			    x1 + 1, y1 + 4);
		p->drawLine(x1 + 5, y1 + 5,
			    x1, y1 + 5 + 5);
		p->drawLine(x1 + 4 + 5, y1 + 1 + 5*2,
			    x1 + 1, y1 + 4 + 5*2);
	    }

	    p->setPen(cg.light());
	    if (slider->orientation() == Qt::Horizontal) {
		x1 = (handle.left() + handle.right()) / 2 - 8;
		y1 = ((handle.top() + handle.bottom()) - 5) / 2;
		p->drawLine(x1 + 4, y1 + 2, 
			    x1 + 2, y1 + 4);
		p->drawLine(x1 + 5 + 5, y1 + 1,
			    x1 + 1 + 5, y1 + 5);
		p->drawLine(x1 + 4 + 5*2, y1 + 2,
			    x1 + 2 + 5*2, y1 + 4);
	    } else {
		x1 = ((handle.left() + handle.right()) - 5) / 2;
		y1 = (handle.top() + handle.bottom()) / 2 - 8;
		p->drawLine(x1 + 4, y1 + 2,
			    x1 + 2, y1 + 4);
		p->drawLine(x1 + 5, y1 + 1 + 5,
			    x1 + 1, y1 + 5 + 5);
		p->drawLine(x1 + 4, y1 + 2 + 5*2,
			    x1 + 2, y1 + 4 + 5*2);
	    }				
	}

	if (controls & SC_SliderTickmarks)
	    QCommonStyle::drawComplexControl(control, p, widget, r, cg, flags, SC_SliderTickmarks, active, data );
	break;
    }

    default:
	QCommonStyle::drawComplexControl(control, p, widget, r, cg, flags,
					 controls, active, data);
	break;
    }
}

QRect BluecurveStyle::querySubControlMetrics( ComplexControl control,
					       const QWidget *widget,
					       SubControl sc,
					       const QStyleOption &data ) const
{
    QRect ret;

    switch (control) {
    case CC_ScrollBar:
    {
	const QScrollBar *scrollbar = (const QScrollBar *) widget;
	int sliderstart = scrollbar->sliderStart();
	int sbextent = pixelMetric(PM_ScrollBarExtent, widget);
	int maxlen = ((scrollbar->orientation() == Qt::Horizontal) ?
		      scrollbar->width() : scrollbar->height()) - (sbextent * 3);
	int sliderlen;

	// calculate slider length
	if (scrollbar->maxValue() != scrollbar->minValue()) {
	    uint range = scrollbar->maxValue() - scrollbar->minValue();
	    sliderlen = (scrollbar->pageStep() * maxlen) /
		(range + scrollbar->pageStep());

	    int slidermin = pixelMetric( PM_ScrollBarSliderMin, widget );
	    if ( sliderlen < slidermin || range > INT_MAX / 2 )
		sliderlen = slidermin;
	    if ( sliderlen > maxlen )
		sliderlen = maxlen;
	} else
	    sliderlen = maxlen;

	switch (sc) {
	case SC_ScrollBarSubLine:
	    // top/left button
	    ret.setRect(0, 0, sbextent, sbextent);
	    break;

	case SC_ScrollBarAddLine:
	    // bottom/right button
	    if (scrollbar->orientation() == Qt::Horizontal)
		ret.setRect(scrollbar->width() - sbextent, 0, sbextent, sbextent);
	    else
		ret.setRect(0, scrollbar->height() - sbextent, sbextent, sbextent);
	    break;

	case SC_ScrollBarSubPage:
	    // between top/left button and slider
	    if (scrollbar->orientation() == Qt::Horizontal)
		ret.setRect(sbextent, 0, sliderstart - sbextent, sbextent);
	    else
		ret.setRect(0, sbextent, sbextent, sliderstart - sbextent);
	    break;

	case SC_ScrollBarAddPage:
	    // between bottom/right button and slider
	    if (scrollbar->orientation() == Qt::Horizontal)
		ret.setRect(sliderstart + sliderlen, 0,
			    maxlen - sliderstart - sliderlen + sbextent * 2, sbextent);
	    else
		ret.setRect(0, sliderstart + sliderlen,
			    sbextent, maxlen - sliderstart - sliderlen + sbextent * 2);
	    break;

	case SC_ScrollBarGroove:
	    if (scrollbar->orientation() == Qt::Horizontal)
		ret.setRect(sbextent, 0, scrollbar->width() - sbextent * 2,
			    scrollbar->height());
	    else
		ret.setRect(0, sbextent, scrollbar->width(),
			    scrollbar->height() - sbextent * 2);
	    break;

	case SC_ScrollBarSlider:
	    if (scrollbar->orientation() == Qt::Horizontal)
		ret.setRect(sliderstart, 0, sliderlen, sbextent);
	    else
		ret.setRect(0, sliderstart, sbextent, sliderlen);
	    break;

	default:
	    break;
	}

	break;
    }

    case CC_SpinWidget:
    {
	int fw = pixelMetric( PM_SpinBoxFrameWidth, widget);
        QSize bs;
        bs.setHeight( widget->height()/2 - fw );
        if ( bs.height() < 8 )
            bs.setHeight( 8 );
        bs.setWidth( bs.height() * 8 / 6 ); 
        bs = bs.expandedTo( QApplication::globalStrut() );
        int y = fw;
        int x, lx, rx;
        x = widget->width() - y - bs.width()+1;
        lx = fw;
        rx = x - fw;
        switch ( sc ) {
        case SC_SpinWidgetUp:
            ret.setRect(x, y-1, bs.width(), bs.height()+1);
            break;
        case SC_SpinWidgetDown:
            ret.setRect(x, y + bs.height(), bs.width(), bs.height()+1);
            break;
        case SC_SpinWidgetButtonField:
            ret.setRect(x, y, bs.width()+1, widget->height() - 2*fw);
            break;
        case SC_SpinWidgetEditField:
            ret.setRect(lx, fw, rx, widget->height() - 2*fw);
            break;
        case SC_SpinWidgetFrame:
            ret = widget->rect();
        default:
            break;
        }

	break;
    }
    default:
	ret = QCommonStyle::querySubControlMetrics(control, widget, sc, data);
	break;
    }

    return ret;
}

QStyle::SubControl BluecurveStyle::querySubControl( ComplexControl control,
						     const QWidget *widget,
						     const QPoint &pos,
						     const QStyleOption &data ) const
{
    QStyle::SubControl ret = QCommonStyle::querySubControl(control, widget, pos, data);

    return ret;
}

int BluecurveStyle::pixelMetric( PixelMetric metric, const QWidget *widget ) const
{
    int ret;

    switch (metric) {
    case PM_ButtonMargin:
	ret = 10;
	break;

    case PM_ButtonShiftHorizontal:
    case PM_ButtonShiftVertical:
	ret = 0;
	break;

    case PM_ButtonDefaultIndicator:
	ret = 1;
	break;
    case PM_DefaultFrameWidth:
	ret = 2;
	break;

    case PM_IndicatorWidth:
    case PM_IndicatorHeight:
    case PM_ExclusiveIndicatorWidth:
    case PM_ExclusiveIndicatorHeight:
	ret = 13;
	break;

    case PM_TabBarTabOverlap:
	ret = 0;
	break;

    case PM_ScrollBarExtent:
	ret = 15;
	break;

    case PM_MenuBarFrameWidth:
	ret = 2;
	break;

    case PM_ProgressBarChunkWidth:
	ret = 2;
	break;

    case PM_DockWindowSeparatorExtent:
	ret = 4;
	break;
    case PM_DockWindowHandleExtent:
	ret = 10;

    case PM_SplitterWidth:
	ret = 8;
	break;

    case PM_ScrollBarSliderMin:
	ret=31;
	break;

    case PM_SliderControlThickness:
	ret = basestyle->pixelMetric( metric, widget );
	break;

    case PM_SliderLength:
    {
	ret=31;
	if (widget->width()<ret)
	    ret = widget->width();
    }
    break;

    case PM_MaximumDragDistance:
	ret = -1;
	break;

    default:
	ret = QCommonStyle::pixelMetric(metric, widget);
	break;
    }

    return ret;
}

QSize BluecurveStyle::sizeFromContents( ContentsType contents,
					 const QWidget *widget,
					 const QSize &contentsSize,
					 const QStyleOption &data ) const
{
    QSize ret;

    switch (contents) {
    case CT_PushButton:
    {
	const QPushButton *button = (const QPushButton *) widget;
	ret = QCommonStyle::sizeFromContents( contents, widget, contentsSize, data );
	int w = ret.width(), h = ret.height();

	// only expand the button if we are displaying text...
	if ( ! button->pixmap() ) {
	    if ( button->isDefault() || button->autoDefault() ) {
		// default button minimum size
		if ( w < 80 )
		    w = 80;
		if ( h < 25 )
		    h = 25;
	    } else {
		// regular button minimum size
		if ( w < 76 )
		    w = 76;
		if ( h < 21 )
		    h = 21;
	    }
	}

	ret = QSize( w, h );
	break;
    }

    case CT_PopupMenuItem:
    {
	if (! widget || data.isDefault())
	    break;

	QMenuItem *mi = data.menuItem();
	const QPopupMenu *popupmenu = (const QPopupMenu *) widget;
	int maxpmw = data.maxIconWidth();
	int w = contentsSize.width(), h = contentsSize.height();

	if (mi->custom()) {
	    w = mi->custom()->sizeHint().width();
	    h = mi->custom()->sizeHint().height();
	    if (! mi->custom()->fullSpan() && h < 22)
		h = 22;
	} else if(mi->widget()) {
	} else if (mi->isSeparator()) {
	    w = 10;
	    h = 4;
	} else {
	    // check is at least 16x16
	    if (h < 16)
		h = 16;
	    if (mi->pixmap())
		h = QMAX(h, mi->pixmap()->height());
	    else if (! mi->text().isNull())
		h = QMAX(h, popupmenu->fontMetrics().height() + 2);
	    if (mi->iconSet() != 0)
		h = QMAX(h, mi->iconSet()->pixmap(QIconSet::Small,
						  QIconSet::Normal).height());
	    h += 2;
	}

	// check | 4 pixels | item | 8 pixels | accel | 4 pixels | check

	// check is at least 16x16
	maxpmw = QMAX(maxpmw, 16);
	w += (maxpmw * 2) + 8;

	if (! mi->text().isNull() && mi->text().find('\t') >= 0)
	    w += 8;

	ret = QSize(w, h);
	break;
    }

    case CT_ComboBox:
    {
	ret = QCommonStyle::sizeFromContents( contents, widget, 
					      contentsSize, data );
	int w = ret.width(), h = ret.height();
	if (h < 27)
	    h = 27;

	ret = QSize(w, h);
	break;
    }

    default:
	ret = QCommonStyle::sizeFromContents(contents, widget, contentsSize, data);
	break;
    }

    return ret;
}

int BluecurveStyle::styleHint( StyleHint stylehint,
				const QWidget *widget,
				const QStyleOption &option,
				QStyleHintReturn* returnData ) const
{
    int ret;

    switch (stylehint) {
    case SH_EtchDisabledText:
    case SH_Slider_SnapToValue:
    case SH_PrintDialog_RightAlignButtons:
    case SH_FontDialog_SelectAssociatedText:
    case SH_MenuBar_AltKeyNavigation:
    case SH_MenuBar_MouseTracking:
    case SH_PopupMenu_MouseTracking:
    case SH_PopupMenu_SpaceActivatesItem:
    case SH_ComboBox_ListMouseTracking:
    case SH_ScrollBar_MiddleClickAbsolutePosition:
	ret = 1;
	break;

    case SH_MainWindow_SpaceBelowMenuBar:
	ret = 0;
	break;

    case SH_PopupMenu_AllowActiveAndDisabled:
	ret = 0;
	break;

    default:
	ret = QCommonStyle::styleHint(stylehint, widget, option, returnData);
	break;
    }

    return ret;
}

QPixmap BluecurveStyle::stylePixmap( StylePixmap stylepixmap,
				      const QWidget *widget,
				      const QStyleOption &data ) const
{
    return basestyle->stylePixmap( stylepixmap, widget, data );
}

void BluecurveStyle::drawGradient(QPainter *p, QRect const &r, QColor const &base, bool horiz, int hdiff, int sdiff, int vdiff) const
{
    QColor c(base);
    p->setPen(c);
    int start, end, left, right, top, bottom;
    if(horiz) {
	start=r.left();
	end=r.right();
	top=r.top();
	bottom=r.bottom();
    } else {
	start=r.top();
	end=r.bottom();
	left=r.left();
	right=r.right();
    }
    int h_, s_, v_;
    c.hsv(&h_, &s_, &v_);
	
    double h(h_);
    double s(s_);
    double v(v_);
    double dh = (start!=end) ? (MIN(hdiff,359-h_)/abs(end-start)) : 0;
    double ds = (start!=end) ? (MIN(sdiff,255-s_)/abs(end-start)) : 0;
    double dv = (start!=end) ? (MIN(vdiff,255-v_)/abs(end-start)) : 0;

    for(int i=start; i<=end; i++) {
	if(horiz)
	    p->drawLine(i, top, i, bottom);
	else
	    p->drawLine(left, i, right, i);
	h += dh;
	s += ds;
	v += dv;
	c.setHsv(int(h), int(s), int(v));
	p->setPen(c);
    }
}
void BluecurveStyle::drawGradientBox(QPainter *p, QRect const &r, QColor const &c, bool horiz, int hdiff, int sdiff, int vdiff) const {
    QRect grad(r.left()+2, r.top()+2, r.width()-3, r.height()-3);
    drawGradient(p, grad, c, horiz, hdiff, sdiff, vdiff);

    // 3d border effect...
    int h, s, v;
    c.hsv(&h, &s, &v);
    QColor tmp(h, s, int(v*.2), QColor::Hsv);
    p->setPen(tmp);
    p->setBrush(NoBrush);
    p->drawRect(r);
    tmp.setHsv(h, s, MIN(255,int(v*1.8)));
    p->setPen(tmp);
    p->drawLine(r.left()+1, r.top()+1, r.right()-1, r.top()+1);
    p->drawLine(r.left()+1, r.top()+1, r.left()+1, r.bottom()-1);
}

// vim:ts=4:sw=4
