/* Credits app */

/* 
 * Copyright (C) 2002 Red Hat, Inc.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <libintl.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>

#define _(x) gettext (x)
#define N_(x) x

struct {
  const char *role;
  const char *name;
  int role_width;
  int name_width;
  PangoLayout *role_layout;
  PangoLayout *name_layout;
  int bold;
} credits[] = {
  
  { "RED HAT LINUX 8.0", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },    
  { "OS DEVELOPMENT", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Producer", "PRESTON BROWN", 0, 0, NULL, NULL, 0 },
  { "Director", "BILL NOTTINGHAM", 0, 0, NULL, NULL, 0 },
  { "Associate Director", "MICHAEL FULBRIGHT", 0, 0, NULL, NULL, 0 },
  { "Cinematographer", "HAVOC PENNINGTON", 0, 0, NULL, NULL, 0 },
  { "Second Assistant Director", "JEREMY KATZ", 0, 0, NULL, NULL, 0 },
  { "Set Constructor", "FLORIAN LA ROCHE", 0, 0, NULL, NULL, 0 },
  { "Composer", "BRENT FOX", 0, 0, NULL, NULL, 0 },
  { "Caterer", "PHIL KNIRSCH", 0, 0, NULL, NULL, 0 },
  { "Terminal Velocity", "NALIN DAHYABHAI", 0, 0, NULL, NULL, 0 },
  { "Boom Operator", "HARALD HOYER", 0, 0, NULL, NULL, 0 },
  { "Standards and Practices", "MATT WILSON", 0, 0, NULL, NULL, 0 },
  { "Kung Fu Choreographer", "THAN NGO", 0, 0, NULL, NULL, 0 },
  { "CG Artist / Programmer", "BERNHARD ROSENKRÄNZER", 0, 0, NULL, NULL, 0 },
  { "Norwegian Møøse Trainer", "TROND EIVIND GLOMSRØD", 0, 0, NULL, NULL, 0 },
  { "Cultural Attaché", "JONATHAN BLANDFORD", 0, 0, NULL, NULL, 0 },
  { "Stunt Double", "KARSTEN HOPP", 0, 0, NULL, NULL, 0 },
  { "Stunt Double", "CHRIS BLIZZARD", 0, 0, NULL, NULL, 0 },
  { "Steadicam Operator", "DAN WALSH", 0, 0, NULL, NULL, 0 },
  { "Miscreant", "TIM WAUGH", 0, 0, NULL, NULL, 0 },
  { "Steely-eyed Linux Man", "OWEN TAYLOR", 0, 0, NULL, NULL, 0 },
  { "Swedish Guy #1", "ALEX LARSSON", 0, 0, NULL, NULL, 0 },
  { "Moose Trainer", "MARK J. COX", 0, 0, NULL, NULL, 0 },
  { "Monkey Handler", "GARY BENSON", 0, 0, NULL, NULL, 0 },
  { "Funeral Director", "JOE ORTON", 0, 0, NULL, NULL, 0 },
  { "Animal Wrangler", "PHILIP COPELAND", 0, 0, NULL, NULL, 0 },
  { "Wardrobe", "ERIK TROAN", 0, 0, NULL, NULL, 0 },
  { "X-Man", "MIKE HARRIS", 0, 0, NULL, NULL, 0 },
  { "Simplified Chinese & Asian X11", "YU SHAO", 0, 0, NULL, NULL, 0 },
  { "Traditional Chinese & Asian KDE", "LEON HO", 0, 0, NULL, NULL, 0 },
  { "Japanese & Asian Compliance", "YUKIHIRO NAKAI", 0, 0, NULL, NULL, 0 },
  { "Japanese & Asian GNOME", "AKIRA TAGOH", 0, 0, NULL, NULL, 0 },
  { "Auto Mechanic", "JENS PETERSEN", 0, 0, NULL, NULL, 0 },
  { "Miracle Worker", "JAKUB JEL\xc3\x8dNEK", 0, 0, NULL, NULL, 0 },
  { "Free Mason", "DAVE", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "KERNEL DEVELOPMENT", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Feline Facilitator", "MICHAEL K. JOHNSON", 0, 0, NULL, NULL, 0 },
  { "Kolonel Patch", "ARJAN VAN DE VEN", 0, 0, NULL, NULL, 0 },
  { "Pointer Casting", "ALAN COX", 0, 0, NULL, NULL, 0 },
  { "Event Co-ordinator", "BEN LaHAISE", 0, 0, NULL, NULL, 0 },
  { "Fortune Cookie Stuffer", "DOUG LEDFORD", 0, 0, NULL, NULL, 0 },
  { "The Teacher", "BOB MATTHEWS", 0, 0, NULL, NULL, 0 },
  { "Networking Entomologist", "DAVID S. MILLER", 0, 0, NULL, NULL, 0 },
  { "Scheduling", "INGO MOLNAR", 0, 0, NULL, NULL, 0 },
  { "International Man of Mystery", "STEPHEN TWEEDIE", 0, 0, NULL, NULL, 0 },
  { "Negative Dentry", "AL VIRO", 0, 0, NULL, NULL, 0 },
  { "USB Cable Wrangler", "PETE ZAITCEV", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },  

  { "TOOLCHAIN DEVELOPMENT", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "General Contractor", "KAREN BENNET", 0, 0, NULL, NULL, 0 },
  { "GCC 3.2 Specialist", "RICHARD HENDERSON", 0, 0, NULL, NULL, 0 },
  { "OProfile Wrangler", "WILL COHEN", 0, 0, NULL, NULL, 0 },
  { "2nd 2nd Assistant Prestidigitator", "ROLAND McGRATH", 0, 0, NULL, NULL, 0 },
  { "Backhoe Operator", "JAKUB JEL\xc3\x8dNEK", 0, 0, NULL, NULL, 0 },
  { "Tools Generalist", "JEFF LAW", 0, 0, NULL, NULL, 0 },
  { "Foley Artist", "ANDREW HALEY", 0, 0, NULL, NULL, 0 },
  { "Ozzy Fan", "JOHN HEALY", 0, 0, NULL, NULL, 0 },
  { "Compiler Tricks", "DIEGO NOVILLO", 0, 0, NULL, NULL, 0 },
  { "GDB Debugging", "JIM BLANDY", 0, 0, NULL, NULL, 0 },
  { "TLS Provider", "ELENA ZANNONI", 0, 0, NULL, NULL, 0 },
  { "Performance Tuner", "VLADIMIR MAKAROV", 0, 0, NULL, NULL, 0 },
  { "Mr. Java", "TOM TROMEY", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "QUALITY ASSURANCE", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Stunt Rigger", "JAY TURNER", 0, 0, NULL, NULL, 0 },
  { "Swing Gang", "BROCK ORGAN",0, 0, NULL, NULL, 0 },
  { "Render Wrangler", "MIKE McLEAN", 0, 0, NULL, NULL, 0 },
  { "Death, Destroyer of Worlds", "BRIAN BROCK", 0, 0, NULL, NULL, 0 },
  { "Digital Message Therapist", "BEN LEVENSON", 0, 0, NULL, NULL, 0 },
  { "Key Grip", "DAVE LAWRENCE", 0, 0, NULL, NULL, 0 },
  { "Korean & Asian Testing", "DAVID JOO", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "RELEASE ENGINEERING", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Benevolent Dictator", "TIM POWERS", 0, 0, NULL, NULL, 0 },
  { "Assistant to Mr. Powers", "JOHN FLANAGAN", 0, 0, NULL, NULL, 0 },
  { "Beekeeper", "ELLIOT LEE", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "DOCUMENTATION", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Content Director", "SANDRA MOORE", 0, 0, NULL, NULL, 0 },
  { "Best Boy Grip", "ED BAILEY", 0, 0, NULL, NULL, 0 },
  { "Script Supervisor", "TAMMY FOX", 0, 0, NULL, NULL, 0 },
  { "Boom Operator", "JOHNRAY FULLER", 0, 0, NULL, NULL, 0 },
  { "Storyboard", "JOHN HA", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "ART DEPARTMENT", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Art Director", "GARRETT LeSAGE", 0, 0, NULL, NULL, 0 },
  { "Sign Painter", "OWEN TAYLOR", 0, 0, NULL, NULL, 0 },
  { "Set Decorators", "THAN NGO", 0, 0, NULL, NULL, 0 },
  { "", "ALEX LARSSON", 0, 0, NULL, NULL, 0 },
  { "", "HAVOC PENNINGTON", 0, 0, NULL, NULL, 0 },
  { "", "BERNHARD ROSENKRÄNZER", 0, 0, NULL, NULL, 0 },
  { "", "PRESTON BROWN", 0, 0, NULL, NULL, 0 },
  { "", "JONATHAN BLANDFORD", 0, 0, NULL, NULL, 0 },
  { "", "MATT WILSON", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "RED HAT NETWORK", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Dictator", "CRISTIAN GAFTON", 0, 0, NULL, NULL, 0 },
  { "Dr. Doom", "ADRIAN LIKINS", 0, 0, NULL, NULL, 0 },
  { "Reverse Engineer", "JEFF JOHNSON", 0, 0, NULL, NULL, 0 },
  { "Prosthetics Technician", "CHIP TURNER", 0, 0, NULL, NULL, 0 },
  { "Patch Dealer", "MIHAI IBANESCU", 0, 0, NULL, NULL, 0 },
  { "Bit Twiddler", "TODD WARNER", 0, 0, NULL, NULL, 0 },
  { "Dolly Grip", "PETER JONES", 0, 0, NULL, NULL, 0 },
  { "XperiMental Linguist", "DANIEL VEILLARD", 0, 0, NULL, NULL, 0 },
  { "Den Mother", "GREG DeKOENIGSBERG", 0, 0, NULL, NULL, 0 },
  { "Minion #2", "BRET McMILLAN", 0, 0, NULL, NULL, 0 },
  { "Special Guest Villain", "ROBIN NORWOOD", 0, 0, NULL, NULL, 0 },
  { "Taint Inspector", "JOSEF KOMENDA", 0, 0, NULL, NULL, 0 },
  { "Political Activist", "BRIAN FAUSTYN", 0, 0, NULL, NULL, 0 },
  { "Paint Technician", "GARRETT LeSAGE", 0, 0, NULL, NULL, 0 },
  { "Cheetah Tamer", "BARRY DUPLANTIS", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },
  
  { "TRANSLATION", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Simplified Chinese", "SARAH SMITH", 0, 0, NULL, NULL, 0 },
  { "Traditional Chinese", "BEN WU", 0, 0, NULL, NULL, 0 },
  { "Czech", "MILAN KERSLAGER", 0, 0, NULL, NULL, 0 },
  { "Danish", "KELD SIMONSEN", 0, 0, NULL, NULL, 0 },
  { "Dutch", "TACO WITTE", 0, 0, NULL, NULL, 0 },
  { "French", "ARNAUD HERVE", 0, 0, NULL, NULL, 0 },
  { "", "VÉRONIQUE MICHEL", 0, 0, NULL, NULL, 0 },
  { "", "EMMANUELLE ROUE", 0, 0, NULL, NULL, 0 },
  { "", "NATHALIE SCHOLZ", 0, 0, NULL, NULL, 0 },
  { "French & Italian", "BETTINA DE MONTI", 0, 0, NULL, NULL, 0 },
  { "Icelandic", "RICHARD ALLEN", 0, 0, NULL, NULL, 0 },
  { "German", "CLAUDIA KRUG", 0, 0, NULL, NULL, 0 },
  { "Greek", "NIKOS CHARONITAKIS", 0, 0, NULL, NULL, 0 },
  { "Korean", "MICHELLE KIM", 0, 0, NULL, NULL, 0 },
  { "Norwegian", "KJARTAN MARAAS", 0, 0, NULL, NULL, 0 },
  { "Portuguese", "PEDRO MORAIS", 0, 0, NULL, NULL, 0 },
  { "Russian", "LEON KANTER", 0, 0, NULL, NULL, 0 },
  { "Spanish", "NURIA SORIANO", 0, 0, NULL, NULL, 0 },
  { "", "LUJÁN FERNÁNDEZ", 0, 0, NULL, NULL, 0 },
  { "Swedish", "CHRISTIAN ROSE", 0, 0, NULL, NULL, 0 },
  { "Turkish", "NILGÜN BELMA BUGÜNER", 0, 0, NULL, NULL, 0 },
  { "Japanese", "JAMES HASHIDA", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "PRODUCT MANAGEMENT", "", 0, 0, NULL, NULL, 1 },
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Choreographer", "SHELLEY BAINTER", 0, 0, NULL, NULL, 0 },
  { "Post-Production Supervisor", "LEIGH DAY", 0, 0, NULL, NULL, 0 },
  { "Speaking In Tongues", "PAUL GAMPE", 0, 0, NULL, NULL, 0 },
  { "Location Manager", "SUE DENHAM", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },

  { "PRODUCTION / FINANCE", "", 0, 0, NULL, NULL, 1},
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Casting", "FEDERICO MUSTO", 0, 0, NULL, NULL, 0 },
  { "Costume Design", "INSA GREENER", 0, 0, NULL, NULL, 0 },
  { "Horse Co-ordinator", "DAVID OWENS", 0, 0, NULL, NULL, 0 },
  { "Promotion Consultant", "MICHAEL PITTMAN", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },
  
  { "SPECIAL CONTRIBUTORS", "", 0, 0, NULL, NULL, 1},
  { "", "", 0, 0, NULL, NULL, 0 },
  { "Himself", "DUMBBOT3", 0, 0, NULL, NULL, 0 },
  { "Crash Test Dummy", "PORKCHOP", 0, 0, NULL, NULL, 0 },
  { "Shover", "KSROBOT", 0, 0, NULL, NULL, 0 },
  { "Pusher", "REROBOT", 0, 0, NULL, NULL, 0 },
  { "", "", 0, 0, NULL, NULL, 0 },
  
  { "SPECIAL THANKS TO", "", 0, 0, NULL, NULL, 1},
  { "", "", 0, 0, NULL, NULL, 0 },
  { "", "Evolution", 0, 0, NULL, NULL, 0 },
  { "", "KDE", 0, 0, NULL, NULL, 0 },
  { "", "GNOME", 0, 0, NULL, NULL, 0 },
  { "", "Mozilla", 0, 0, NULL, NULL, 0 },
  { "", "OpenOffice.org", 0, 0, NULL, NULL, 0 },
  { "", "XFree86", 0, 0, NULL, NULL, 0 },
  { "", "Linus Torvalds", 0, 0, NULL, NULL, 0 },
  { "", "the Linux kernel developers", 0, 0, NULL, NULL, 0 },
  { "", "Apache", 0, 0, NULL, NULL, 0 },
  { "", "Samba", 0, 0, NULL, NULL, 0 },
  { "", "GNU Project", 0, 0, NULL, NULL, 0 },
  { "", "our Beta Testers", 0, 0, NULL, NULL, 0 },
  { "", "the Open Source Community", 0, 0, NULL, NULL, 0 },
};

static int n_credits = G_N_ELEMENTS (credits);  

static GTimeVal start_time;

/* current position in the credits list */
static double current_fraction = 0.0;

/* how long it takes for a line to cross the screen */
static double seconds_to_cross_screen = 3.0;

/* height of one line */
static int line_height = 0;

static int frames_drawn = 0;

static GdkPixbuf *logo = NULL;

static gboolean destroyed = FALSE;

#define MAX_FPS 30
#define MAX_FRAMERATE (1000 /* 1 sec */ / MAX_FPS /* fps */)

static gboolean
timeout (void *data)
{
  GtkWidget *widget;
  GTimeVal current_time;
  double elapsed;
  double total;

  if (destroyed)
    return FALSE;
  
  widget = GTK_WIDGET (data);
  
  g_get_current_time (&current_time);

  elapsed =
    ((((double)current_time.tv_sec - start_time.tv_sec) * G_USEC_PER_SEC +
      (current_time.tv_usec - start_time.tv_usec)));

  total = n_credits * seconds_to_cross_screen * G_USEC_PER_SEC;
  
  elapsed = ((int) elapsed) % ((int) total);

  current_fraction = (elapsed / total) * 2.7;
  
  gtk_widget_queue_draw (widget);

  g_timeout_add (MAX_FRAMERATE, timeout, widget);

  ++frames_drawn;
  
  /* remove this one */
  return FALSE;
}

static gboolean
expose_event (GtkWidget      *widget,
              GdkEventExpose *event,
              void           *data)
{
  int screen_height;
  int screen_width;
  int screen_mid;
  int total_height;
  int total_delta;
  int x_offset;
  int i;
  int spacing;
  GdkGC *gc;
  GdkColor color;

  if (destroyed)
    return FALSE;
  
  total_height = line_height * n_credits;
  
  /* The topmost line moves from being at the bottom of the screen
   * at fraction 0.0, to being at -total_height at fraction 1.0
   *
   * So the position of this line is between
   * [-total_height, screen_height - line_height]
   *
   * total_delta = screen_height - line_height + total_height
   */
  screen_width = gdk_screen_width ();
  screen_height = gdk_screen_height ();
  total_delta = screen_height - line_height + total_height;
 
  spacing = 0.03 * screen_width;
  screen_mid = screen_width / 2;

  gc = gdk_gc_new (widget->window);
  
  i = 0;
  while (i < n_credits)
    {
      int line_y;

      line_y = line_height * i - total_height + total_delta * (1.0 - current_fraction);

      if (line_y >= 0 &&
          line_y < screen_height)
        {
          /* Horizontally, we slide in from left on the bottom, and
           * slide out to right on the top
           */          
          double screen_pos = (double) line_y / (double) screen_height;
          double top_slide_margin = ((double) (line_height * 0.8) / (double) screen_height);
          double bottom_slide_margin = (((double) screen_height - (line_height * 1.5)) / (double) screen_height);

          color.red = 65535;
          color.green = 65535;
          color.blue = 65535;
          
          if (screen_pos < top_slide_margin)
            {
              double fraction;

              fraction = (1.0 - (screen_pos / top_slide_margin));
              
              /* out to right, x_offset from [0, screen_width - left_margin] */
              x_offset = screen_width * fraction;

              color.red *= 1.0 - fraction;
              color.blue *= 1.0 - fraction;
              color.green *= 1.0 - fraction;
            }
          else if (screen_pos > bottom_slide_margin)
            {
              double fraction;

              fraction = (1.0 - (1.0 - screen_pos) / (1.0 - bottom_slide_margin));
	      x_offset = (- screen_width) * fraction;
                
              color.red *= 1.0 - fraction;
              color.blue *= 1.0 - fraction;
              color.green *= 1.0 - fraction;
            }
          else
            x_offset = 0;          
          
          gdk_gc_set_rgb_fg_color (gc, &color);
          
	  if (credits[i].name_width == 0)
	    {
	      gdk_draw_layout(widget->window,
			      gc,
			      x_offset + 
			      (screen_mid - (credits[i].role_width / 2)),
			      line_y,
			      credits[i].role_layout);
					  
	  } 
	  else 
	    {
	      gdk_draw_layout (widget->window,
			       gc,
			       x_offset +
			       (screen_mid - (spacing / 2)) - 
			       credits[i].role_width,
			       line_y,
			       credits[i].role_layout);
	      
	      gdk_draw_layout (widget->window,
			       gc,
			       x_offset +
			       screen_mid + (spacing / 2),
			       line_y,
			       credits[i].name_layout);
	    }
        }
      
      ++i;
    }

  if (logo)
    {
      int w = gdk_pixbuf_get_width (logo);
      int h = gdk_pixbuf_get_height (logo);

      gdk_pixbuf_render_to_drawable (logo,
                                     widget->window,
                                     gc,
                                     0, 0,
                                     screen_width - w - 110,
                                     screen_height - h - 60,
                                     w, h, GDK_RGB_DITHER_MAX,
                                     0, 0);
    }
  
  g_object_unref (G_OBJECT (gc));
  
  return FALSE;
}

static gboolean
button_press_event (GtkWidget      *widget,
                    GdkEventButton *event,
                    void           *data)
{
  if (destroyed)
    return FALSE;
  
  gtk_widget_destroy (widget);

  gtk_main_quit ();
  
  return FALSE;
}

static gboolean
key_press_event (GtkWidget      *widget,
                 GdkEventKey    *event,
                 void           *data)
{
  if (destroyed)
    return FALSE;
  
  gtk_widget_destroy (widget);

  gtk_main_quit ();
  
  return FALSE;
}

static void
destroy_callback (GtkWidget *widget)
{
  destroyed = TRUE;
}

int
main (int argc, char **argv)
{
  GdkColor color;
  GtkWidget *window;
  int i;
  PangoFontDescription *role_font_desc, *name_font_desc;
  double elapsed;
  GTimeVal current_time;
  GError *err;
  int base_font_size;
  int lines_per_screen;
  pid_t pid = 0;
  
  gtk_init (&argc, &argv);  

  if ((access("/usr/bin/ogg123", X_OK) != -1) &&
      (access("/usr/share/redhat-credits.ogg", R_OK) != -1)) {
      pid = fork();
      if (pid == 0) {
	  close(1);
	  close(2);
	  execl("/usr/bin/ogg123", "ogg123", "/usr/share/redhat-credits.ogg", NULL);
      }
  } else {
      g_printerr("Credits theme is missing or ogg123 not found.\n");
  }
  
  err = NULL;
  logo = gdk_pixbuf_new_from_file ("/usr/share/pixmaps/redhat-credits.png",
                                   &err);
  if (err != NULL)
    {
      g_printerr ("Credits logo problem: %s.\n",
                  err->message);
      g_error_free (err);
    }
  
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);

  g_signal_connect (G_OBJECT (window), "destroy",
                    G_CALLBACK (destroy_callback),
                    NULL);
  
  gtk_window_set_decorated (GTK_WINDOW (window), FALSE);
  gtk_widget_set_app_paintable (window, TRUE);
  gtk_window_set_resizable (GTK_WINDOW (window), FALSE);
  gtk_widget_set_size_request (window,
                               gdk_screen_width (),
                               gdk_screen_height ());
  
  color.red = color.blue = color.green = 0;
  gtk_widget_modify_bg (window, GTK_STATE_NORMAL, &color);
  
  gtk_widget_add_events (window, GDK_BUTTON_PRESS_MASK);
  
  g_signal_connect (G_OBJECT (window), "expose_event",
                    G_CALLBACK (expose_event), NULL);

  g_signal_connect (G_OBJECT (window), "button_press_event",
                    G_CALLBACK (button_press_event), NULL);

  g_signal_connect (G_OBJECT (window), "key_press_event",
                    G_CALLBACK (key_press_event), NULL);

  if (gdk_screen_height () >= 1200)
      lines_per_screen = 45;
  else if (gdk_screen_height () >= 1050)
      lines_per_screen = 40;
  else if (gdk_screen_height () >= 1024)
      lines_per_screen = 38;
  else if (gdk_screen_height () >= 768)
      lines_per_screen = 35;
  else
      lines_per_screen = 30;
  
  base_font_size = (gdk_screen_height () / lines_per_screen) * PANGO_SCALE;

  role_font_desc = pango_font_description_new ();
  name_font_desc = pango_font_description_new ();
  pango_font_description_set_family (role_font_desc, "Sans-serif");
  pango_font_description_set_family (name_font_desc, "Sans-serif");
  
  line_height = 0;
  i = 0;
  while (i < n_credits)
    {
      int w, h;
      
      credits[i].role_layout =
        gtk_widget_create_pango_layout (window, credits[i].role);

      if (credits[i].bold) 
	{
	  pango_font_description_set_weight (role_font_desc, 
					     PANGO_WEIGHT_BOLD);
	  pango_font_description_set_size  (role_font_desc,
                                            base_font_size * 
					    PANGO_SCALE_X_LARGE);
	}
      else
        {
	  pango_font_description_set_weight (role_font_desc, 
					     PANGO_WEIGHT_NORMAL);
	  pango_font_description_set_size (role_font_desc, base_font_size);
	}

      pango_layout_set_font_description (credits[i].role_layout,
                                         role_font_desc);
      
      pango_layout_get_pixel_size (credits[i].role_layout,
                                   &w, &h);

      line_height = MAX (line_height, h);
      credits[i].role_width = w;
      
      credits[i].name_layout =
        gtk_widget_create_pango_layout (window, credits[i].name);

      pango_font_description_set_weight (name_font_desc, PANGO_WEIGHT_BOLD);
      pango_font_description_set_size (name_font_desc, base_font_size);
      pango_layout_set_font_description (credits[i].name_layout,
                                         name_font_desc);
      
      pango_layout_get_pixel_size (credits[i].name_layout,
                                   &w, &h);

      line_height = MAX (line_height, h);
      credits[i].name_width = w;
      
      ++i;
    }

  pango_font_description_free (role_font_desc);
  pango_font_description_free (name_font_desc);
  
  g_timeout_add (MAX_FRAMERATE, timeout, window);
  
  g_get_current_time (&start_time);
  
  gtk_widget_show (window);

  gtk_main ();

  if (pid) {
      kill(pid, SIGINT);
      waitpid(pid, NULL, 0);
  }
  
  g_get_current_time (&current_time);
  
  elapsed =
    ((((double)current_time.tv_sec - start_time.tv_sec) * G_USEC_PER_SEC +
      (current_time.tv_usec - start_time.tv_usec)));
  
  /*g_print ("%d frames in %g seconds %g fps with fps capped at %g\n",
           frames_drawn, elapsed / G_USEC_PER_SEC,
           frames_drawn / (elapsed / G_USEC_PER_SEC),
           (double) MAX_FPS);*/
  
  return 0;
}
