#include <gmodule.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include "bluecurve1.h"

#include "../Bluecurve/bits.c"

/* #define DEBUG 1 */

extern GtkStyleClass bluecurve_default_class;

enum
{
  TOKEN_SPOTCOLOR = G_TOKEN_LAST + 1,
  TOKEN_CONTRAST,
  TOKEN_WIDE,

  TOKEN_TRUE,
  TOKEN_FALSE
};

static struct
  {
    const gchar        *name;
    guint               token;
  }
theme_symbols[] =
{
  { "spotcolor", 		TOKEN_SPOTCOLOR  },
  { "contrast", 		TOKEN_CONTRAST  },
  { "wide", 		        TOKEN_WIDE  },
  
  { "TRUE",                TOKEN_TRUE },
  { "FALSE",               TOKEN_FALSE },
};

static guint n_theme_symbols = sizeof(theme_symbols) / sizeof(theme_symbols[0]);

static void
bluecurve_rc_data_ref (BluecurveRcData *bluecurve_data)
{
  bluecurve_data->refcount++;
}

static void
bluecurve_rc_data_unref (BluecurveRcData *bluecurve_data)
{
  bluecurve_data->refcount--;
  if (bluecurve_data->refcount == 0)
    {
      g_free (bluecurve_data);
    }
}

static guint
theme_parse_color(GScanner     *scanner,
		  GdkColor     *color)
{
  guint token;

  /* Skip 'blah_color' */
  token = g_scanner_get_next_token(scanner);

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  return gtk_rc_parse_color (scanner, color);
}

static guint
theme_parse_contrast(GScanner     *scanner,
		     double       *contrast)
{
  guint token;

  /* Skip 'contrast' */
  token = g_scanner_get_next_token(scanner);

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;

  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_FLOAT)
    return G_TOKEN_FLOAT;

  *contrast = scanner->value.v_float;
  
  return G_TOKEN_NONE;
}

static guint
theme_parse_boolean(GScanner *scanner,
                    GTokenType wanted_token,
                    guint *retval)
{
  guint token;
  
  token = g_scanner_get_next_token(scanner);
  if (token != wanted_token)
    return wanted_token;
  
  token = g_scanner_get_next_token(scanner);
  if (token != G_TOKEN_EQUAL_SIGN)
    return G_TOKEN_EQUAL_SIGN;
  
  token = g_scanner_get_next_token(scanner);
  if (token == TOKEN_TRUE)
    *retval = TRUE;
  else if (token == TOKEN_FALSE)
    *retval = FALSE;
  else
    return TOKEN_TRUE;

  return G_TOKEN_NONE;
}

static guint
theme_parse_rc_style (GScanner   *scanner,
		      GtkRcStyle *rc_style)
		     
{
  static GQuark scope_id = 0;
  BluecurveRcData *rc_data;
  guint old_scope;
  guint token;
  guint i;
  
  /* Set up a new scope in this scanner. */

  if (!scope_id)
    scope_id = g_quark_from_string("bluecurve_theme_engine");

  /* If we bail out due to errors, we *don't* reset the scope, so the
   * error messaging code can make sense of our tokens.
   */
  old_scope = g_scanner_set_scope(scanner, scope_id);

  /* Now check if we already added our symbols to this scope
   * (in some previous call to bluecurve_rc_style_parse for the
   * same scanner.
   */

  if (!g_scanner_lookup_symbol(scanner, theme_symbols[0].name))
    {
      g_scanner_freeze_symbol_table(scanner);
      for (i = 0; i < n_theme_symbols; i++)
	g_scanner_scope_add_symbol(scanner, scope_id,
				   theme_symbols[i].name,
				   GINT_TO_POINTER(theme_symbols[i].token));
      g_scanner_thaw_symbol_table(scanner);
    }

  /* We're ready to go, now parse the top level */

  rc_data = g_new(BluecurveRcData, 1);
  rc_data->refcount = 1;
  rc_data->has_spot_color = FALSE;
  rc_data->contrast = 1.0;
  rc_data->wide = 0;

  token = g_scanner_peek_next_token(scanner);
  while (token != G_TOKEN_RIGHT_CURLY)
    {
      switch (token)
	{
	case TOKEN_SPOTCOLOR:
	  token = theme_parse_color(scanner, &rc_data->spot_color);
	  rc_data->has_spot_color = TRUE;
	  break;
	case TOKEN_CONTRAST:
	  token = theme_parse_contrast(scanner, &rc_data->contrast);
	  break;
	case TOKEN_WIDE:
	  token = theme_parse_boolean(scanner, TOKEN_WIDE, &rc_data->wide);
	  break;
	default:
	  g_scanner_get_next_token(scanner);
	  token = G_TOKEN_RIGHT_CURLY;
	  break;
	}

      if (token != G_TOKEN_NONE)
	return token;

      token = g_scanner_peek_next_token(scanner);
    }

  g_scanner_get_next_token(scanner);

  rc_style->engine_data = rc_data;
  g_scanner_set_scope(scanner, old_scope);

  return G_TOKEN_NONE;
}

static void
theme_merge_rc_style (GtkRcStyle *dest,
		      GtkRcStyle *src)
{
  BluecurveRcData *src_data = src->engine_data;
  
  if (!dest->engine_data)
    {
      if (src_data)
	{
	  bluecurve_rc_data_ref (src_data);
	  dest->engine_data = src_data;
	}
    }
}

static void
set_props (GtkStyle *style)
{
  /* TODO: Change and/or add to these: */
#define OPTION_INDICATOR_WIDTH 7
#define OPTION_INDICATOR_LEFT_SPACING 7
#define OPTION_INDICATOR_RIGHT_SPACING 5
  gtk_style_set_prop_experimental (style, "GtkButton::default_spacing", 6);
  gtk_style_set_prop_experimental (style, "GtkCheckButton::indicator_size", 13);
  gtk_style_set_prop_experimental (style, "GtkOptionMenu::indicator_width", OPTION_INDICATOR_WIDTH);
  gtk_style_set_prop_experimental (style, "GtkOptionMenu::indicator_left_spacing", OPTION_INDICATOR_LEFT_SPACING);
  gtk_style_set_prop_experimental (style, "GtkOptionMenu::indicator_right_spacing", OPTION_INDICATOR_RIGHT_SPACING);
  gtk_style_set_prop_experimental (style, "GtkPaned::handle_full_size", 1);
  gtk_style_set_prop_experimental (style, "GtkPaned::handle_width", 6);
  gtk_style_set_prop_experimental (style, "GtkRange::always_draw_trough", 1);
  gtk_style_set_prop_experimental (style, "GtkRange::trough_border", 0);
  gtk_style_set_prop_experimental (style, "GtkRange::slider_width", 15);
  gtk_style_set_prop_experimental (style, "GtkRange::stepper_size", 15);
  gtk_style_set_prop_experimental (style, "GtkRange::stepper_spacing", 0);
  gtk_style_set_prop_experimental (style, "GtkSpinButton::shadow_type", GTK_SHADOW_IN);
}

static GdkColor *
bluecurve_get_spot_color (GtkRcStyle *rc_style)
{
  BluecurveRcData *rc_data = rc_style->engine_data;
  
  if (rc_data->has_spot_color)
    return &rc_data->spot_color;
  else
    return &rc_style->base[GTK_STATE_SELECTED];
}

/****************************************************************************/

static void
rgb_to_hls (gdouble *r,
            gdouble *g,
            gdouble *b)
{
  gdouble min;
  gdouble max;
  gdouble red;
  gdouble green;
  gdouble blue;
  gdouble h, l, s;
  gdouble delta;
  
  red = *r;
  green = *g;
  blue = *b;
  
  if (red > green)
    {
      if (red > blue)
        max = red;
      else
        max = blue;
      
      if (green < blue)
        min = green;
      else
        min = blue;
    }
  else
    {
      if (green > blue)
        max = green;
      else
        max = blue;
      
      if (red < blue)
        min = red;
      else
        min = blue;
    }
  
  l = (max + min) / 2;
  s = 0;
  h = 0;
  
  if (max != min)
    {
      if (l <= 0.5)
        s = (max - min) / (max + min);
      else
        s = (max - min) / (2 - max - min);
      
      delta = max -min;
      if (red == max)
        h = (green - blue) / delta;
      else if (green == max)
        h = 2 + (blue - red) / delta;
      else if (blue == max)
        h = 4 + (red - green) / delta;
      
      h *= 60;
      if (h < 0.0)
        h += 360;
    }
  
  *r = h;
  *g = l;
  *b = s;
}

static void
hls_to_rgb (gdouble *h,
            gdouble *l,
            gdouble *s)
{
  gdouble hue;
  gdouble lightness;
  gdouble saturation;
  gdouble m1, m2;
  gdouble r, g, b;
  
  lightness = *l;
  saturation = *s;
  
  if (lightness <= 0.5)
    m2 = lightness * (1 + saturation);
  else
    m2 = lightness + saturation - lightness * saturation;
  m1 = 2 * lightness - m2;
  
  if (saturation == 0)
    {
      *h = lightness;
      *l = lightness;
      *s = lightness;
    }
  else
    {
      hue = *h + 120;
      while (hue > 360)
        hue -= 360;
      while (hue < 0)
        hue += 360;
      
      if (hue < 60)
        r = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
        r = m2;
      else if (hue < 240)
        r = m1 + (m2 - m1) * (240 - hue) / 60;
      else
        r = m1;
      
      hue = *h;
      while (hue > 360)
        hue -= 360;
      while (hue < 0)
        hue += 360;
      
      if (hue < 60)
        g = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
        g = m2;
      else if (hue < 240)
        g = m1 + (m2 - m1) * (240 - hue) / 60;
      else
        g = m1;
      
      hue = *h - 120;
      while (hue > 360)
        hue -= 360;
      while (hue < 0)
        hue += 360;
      
      if (hue < 60)
        b = m1 + (m2 - m1) * hue / 60;
      else if (hue < 180)
        b = m2;
      else if (hue < 240)
        b = m1 + (m2 - m1) * (240 - hue) / 60;
      else
        b = m1;
      
      *h = r;
      *l = g;
      *s = b;
    }
}

void
bluecurve_shade (GdkColor * a, GdkColor * b, float k)
{
  gdouble red;
  gdouble green;
  gdouble blue;
  
  red = (gdouble) a->red / 65535.0;
  green = (gdouble) a->green / 65535.0;
  blue = (gdouble) a->blue / 65535.0;
  
  rgb_to_hls (&red, &green, &blue);
  
  green *= k;
  if (green > 1.0)
    green = 1.0;
  else if (green < 0.0)
    green = 0.0;
  
  blue *= k;
  if (blue > 1.0)
    blue = 1.0;
  else if (blue < 0.0)
    blue = 0.0;
  
  hls_to_rgb (&red, &green, &blue);
  
  b->red = red * 65535.0;
  b->green = green * 65535.0;
  b->blue = blue * 65535.0;
}

static void
shade (GdkColor * a, GdkColor * b, float k)
{
  bluecurve_shade (a, b, k);
}

/**************************************************************************/

static void
theme_rc_style_to_style(GtkStyle * style,
			GtkRcStyle * rc_style)
{
  static GtkStyleClass *class = NULL;
  static GtkStyleClass *wide_class = NULL;
  BluecurveRcData *rc_data = rc_style->engine_data;
  BluecurveData *bluecurve_data;
  GdkColor *spot_color;
  double shades[] = {1.065, 0.963, 0.896, 0.85, 0.768, 0.665, 0.4, 0.205};
  int i;
  double contrast;

  if (!class)
    {
      GtkStyle *tmp_style = gtk_style_new ();
      class = g_new (GtkStyleClass, 1);

      bluecurve_initialize_style (class, style->klass);
      gtk_style_unref (tmp_style);
      class->xthickness = 1;
      class->ythickness = 1;
    }
  
  if (!wide_class)
    {
      GtkStyle *tmp_style = gtk_style_new ();
      wide_class = g_new (GtkStyleClass, 1);

      bluecurve_initialize_style (wide_class, style->klass);
      gtk_style_unref (tmp_style);
      class->xthickness = 2;
      class->ythickness = 2;
    }

  if (rc_data->wide)
    style->klass = wide_class;
  else
    style->klass = class;
  bluecurve_data = style->engine_data = g_new0 (BluecurveData, 1);

  contrast = rc_data->contrast;

  /* Lighter to darker */
  for (i = 0; i < 8; i++)
    shade (&style->bg[GTK_STATE_NORMAL], &bluecurve_data->gray[i], (shades[i]-0.7) * contrast + 0.7);
  
  spot_color = bluecurve_get_spot_color (rc_style);

  bluecurve_data->spot_color = *spot_color;
  shade (&bluecurve_data->spot_color, &bluecurve_data->spot1, 1.62);
  shade (&bluecurve_data->spot_color, &bluecurve_data->spot2, 1.05);
  shade (&bluecurve_data->spot_color, &bluecurve_data->spot3, 0.72);
  
  set_props (style);
}

static void
theme_duplicate_style(GtkStyle * dest,
		      GtkStyle * src)
{
  dest->engine_data = g_memdup (src->engine_data, sizeof (BluecurveData));
  
  set_props (dest);
}

static GdkGC *
realize_color (GtkStyle * style,
	       GdkColor * color)
{
  GdkGCValues gc_values;

  gdk_colormap_alloc_color (style->colormap, color,
			    FALSE, TRUE);

  gc_values.foreground = *color;

  return gtk_gc_get (style->depth, style->colormap,
		     &gc_values, GDK_GC_FOREGROUND);
}

static GdkPixbuf *
generate_bit (unsigned char alpha[], GdkColor *color, double mult)
{
  guint r, g, b;
  GdkPixbuf *pixbuf;
  unsigned char *pixels;
  int w, h, rs;
  int x, y;
  
  r = (color->red >> 8) * mult;
  r = MIN(r, 255);
  g = (color->green >> 8) * mult;
  g = MIN(g, 255);
  b = (color->blue >> 8) * mult;
  b = MIN(b, 255);

  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, RADIO_SIZE, RADIO_SIZE);

  w = gdk_pixbuf_get_width (pixbuf);
  h = gdk_pixbuf_get_height (pixbuf);
  rs = gdk_pixbuf_get_rowstride (pixbuf);
  pixels = gdk_pixbuf_get_pixels (pixbuf);


  for (y=0; y < h; y++)
    {
      for (x=0; x < w; x++)
	{
	  pixels[y*rs + x*4 + 0] = r;
	  pixels[y*rs + x*4 + 1] = g;
	  pixels[y*rs + x*4 + 2] = b;
	  if (alpha)
	    pixels[y*rs + x*4 + 3] = alpha[y*w + x];
	  else
	    pixels[y*rs + x*4 + 3] = 255;
	}
    }

  return pixbuf;
}

#define CLAMP_UCHAR(v) ((guchar) (CLAMP (((int)v), (int)0, (int)255)))

static GdkPixbuf *
colorize_bit (unsigned char *bit,
	      unsigned char *alpha,
	      GdkColor  *new_color)
{
  GdkPixbuf *pixbuf;
  double intensity;
  int x, y;
  const guchar *src, *asrc;
  guchar *dest;
  int dest_rowstride;
  int width, height;
  guchar *dest_pixels;
  
  pixbuf = gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8,
			   RADIO_SIZE, RADIO_SIZE);

  if (pixbuf == NULL)
    return NULL;
  
  dest_rowstride = gdk_pixbuf_get_rowstride (pixbuf);
  width = gdk_pixbuf_get_width (pixbuf);
  height = gdk_pixbuf_get_height (pixbuf);
  dest_pixels = gdk_pixbuf_get_pixels (pixbuf);
  
  for (y = 0; y < RADIO_SIZE; y++)
    {
      src = bit + y * RADIO_SIZE;
      asrc = alpha + y * RADIO_SIZE;
      dest = dest_pixels + y * dest_rowstride;

      for (x = 0; x < RADIO_SIZE; x++)
        {
          double dr, dg, db;
          
          intensity = (src[x] + 0 )/ 255.0;

          if (intensity <= 0.5)
            {
              /* Go from black at intensity = 0.0 to new_color at intensity = 0.5 */
              dr = (new_color->red * intensity * 2.0) / 65535.0;
              dg = (new_color->green * intensity * 2.0) / 65535.0;
              db = (new_color->blue * intensity * 2.0) / 65535.0;
            }
          else
            {
              /* Go from new_color at intensity = 0.5 to white at intensity = 1.0 */
              dr = (new_color->red + (65535 - new_color->red) * (intensity - 0.5) * 2.0) / 65535.0;
              dg = (new_color->green + (65535 - new_color->green) * (intensity - 0.5) * 2.0) / 65535.0;
              db = (new_color->blue + (65535 - new_color->blue) * (intensity - 0.5) * 2.0) / 65535.0;
            }
          
          dest[0] = CLAMP_UCHAR (255 * dr);
          dest[1] = CLAMP_UCHAR (255 * dg);
          dest[2] = CLAMP_UCHAR (255 * db);

	  dest[3] = asrc[x];
	  dest += 4;
        }
    }

  return pixbuf;
}

static GdkPixmap *
pixbuf_to_pixmap (GtkStyle * style, GdkPixbuf *pixbuf)
{
  GdkGC *tmp_gc;
  GdkPixmap *pixmap;
  
  pixmap = gdk_pixmap_new (NULL,
			   gdk_pixbuf_get_width (pixbuf),
			   gdk_pixbuf_get_height (pixbuf),
			   style->depth);
  /*gdk_drawable_set_colormap (pixmap, style->colormap); */

  tmp_gc = gdk_gc_new (pixmap);
  
  gdk_pixbuf_render_to_drawable (pixbuf, pixmap, tmp_gc, 0, 0, 0, 0,
				 gdk_pixbuf_get_width (pixbuf),
				 gdk_pixbuf_get_height (pixbuf),
				 GDK_RGB_DITHER_NORMAL, 0, 0);
  gdk_gc_unref (tmp_gc);

  return pixmap;
}

static void
theme_realize_style(GtkStyle * style)
{
  BluecurveData *bluecurve_data = style->engine_data;
  GdkPixbuf *dot, *circle, *outline, *inconsistent, *composite;
  GdkPixbuf *check, *base;
  GdkColor *spot_color = bluecurve_get_spot_color (style->rc_style);
  GdkColor *composite_color;
  int i;

  for (i = 0; i < 8; i++)
    bluecurve_data->gray_gc[i] = realize_color (style, &bluecurve_data->gray[i]);

  bluecurve_data->spot1_gc = realize_color (style, &bluecurve_data->spot1);
  bluecurve_data->spot2_gc = realize_color (style, &bluecurve_data->spot2);
  bluecurve_data->spot3_gc = realize_color (style, &bluecurve_data->spot3);

  dot = colorize_bit (dot_intensity, dot_alpha, spot_color);
  inconsistent = generate_bit (inconsistent_alpha, spot_color, 1.0);
  outline = generate_bit (outline_alpha, &bluecurve_data->gray[6], 1.0);

  gdk_pixbuf_render_pixmap_and_mask (outline,
				     NULL,
				     &bluecurve_data->radio_pixmap_mask,
				     1);
  
  for (i = 0; i < 5; i++)
    {
      if (i == GTK_STATE_ACTIVE)
	{
	  composite_color = &style->bg[GTK_STATE_PRELIGHT];
	  circle = generate_bit (circle_alpha, &style->bg[i], 1.0);
	}
      else
	{
	  composite_color = &style->bg[i];
	  circle = generate_bit (circle_alpha, &style->white, 1.0);
	}

      composite = generate_bit (NULL, composite_color, 1.0);

      gdk_pixbuf_composite (outline, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      gdk_pixbuf_composite (circle, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);

      bluecurve_data->radio_pixmap_nonactive[i] = pixbuf_to_pixmap (style, composite);
      
      gdk_pixbuf_composite (dot, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      
      bluecurve_data->radio_pixmap_active[i] = pixbuf_to_pixmap (style, composite);
      
      gdk_pixbuf_unref (composite);

      composite = generate_bit (NULL, composite_color,1.0);

      gdk_pixbuf_composite (outline, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      gdk_pixbuf_composite (circle, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      gdk_pixbuf_composite (inconsistent, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      
      bluecurve_data->radio_pixmap_inconsistent[i] = pixbuf_to_pixmap (style, composite);
      
      gdk_pixbuf_unref (composite);
      gdk_pixbuf_unref (circle);
    }
  
  gdk_pixbuf_unref (dot);
  gdk_pixbuf_unref (inconsistent);
  gdk_pixbuf_unref (outline);


  check = generate_bit (check_alpha, spot_color, 1.0);
  inconsistent = generate_bit (check_inconsistent_alpha, spot_color, 1.0);

  for (i = 0; i < 5; i++)
    {
      base = generate_bit (check_base_alpha, &style->white, 1.0);
      composite = generate_bit (NULL, &bluecurve_data->gray[6], 1.0);

      gdk_pixbuf_composite (base, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      
      bluecurve_data->check_pixmap_nonactive[i] = pixbuf_to_pixmap (style, composite);
      
      gdk_pixbuf_composite (check, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      
      bluecurve_data->check_pixmap_active[i] = pixbuf_to_pixmap (style, composite);
      
      gdk_pixbuf_unref (composite);

      composite = generate_bit (NULL, &bluecurve_data->gray[6], 1.0);

      gdk_pixbuf_composite (base, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      gdk_pixbuf_composite (inconsistent, composite,
			    0, 0, RADIO_SIZE, RADIO_SIZE, 0, 0,
			    1.0, 1.0, GDK_INTERP_NEAREST, 255);
      
      bluecurve_data->check_pixmap_inconsistent[i] = pixbuf_to_pixmap (style, composite);
      
      gdk_pixbuf_unref (composite);
      gdk_pixbuf_unref (base);
    }
  
  gdk_pixbuf_unref (check);
  gdk_pixbuf_unref (inconsistent);
}

static void
theme_unrealize_style(GtkStyle * style)
{
  BluecurveData *bluecurve_data = style->engine_data;
  int i;
  
  /* We don't free the colors, because we don't know if
   * gtk_gc_release() actually freed the GC. FIXME - need
   * a way of ref'ing colors explicitely so GtkGC can
   * handle things properly.
   */
  for (i=0; i < 8; i++)
    gtk_gc_release (bluecurve_data->gray_gc[i]);

  
  gtk_gc_release (bluecurve_data->spot1_gc);
  gtk_gc_release (bluecurve_data->spot2_gc);
  gtk_gc_release (bluecurve_data->spot3_gc);


  for (i = 0; i < 5; i++)
    {
      gdk_pixmap_unref (bluecurve_data->radio_pixmap_nonactive[i]);
      gdk_pixmap_unref (bluecurve_data->radio_pixmap_active[i]);
      gdk_pixmap_unref (bluecurve_data->radio_pixmap_inconsistent[i]);
      gdk_pixmap_unref (bluecurve_data->check_pixmap_nonactive[i]);
      gdk_pixmap_unref (bluecurve_data->check_pixmap_active[i]);
      gdk_pixmap_unref (bluecurve_data->check_pixmap_inconsistent[i]);
    }
  gdk_pixmap_unref (bluecurve_data->radio_pixmap_mask);
}

static void
theme_destroy_rc_style(GtkRcStyle * rc_style)
{
  BluecurveRcData *rc_data = rc_style->engine_data;
  bluecurve_rc_data_unref (rc_data);
}

static void
theme_destroy_style(GtkStyle * style)
{
  g_free (style->engine_data);
}

G_MODULE_EXPORT void
theme_init (GtkThemeEngine * engine)
{
  engine->parse_rc_style = theme_parse_rc_style;
  engine->merge_rc_style = theme_merge_rc_style;
  engine->rc_style_to_style = theme_rc_style_to_style;
  engine->duplicate_style = theme_duplicate_style;
  engine->realize_style = theme_realize_style;
  engine->unrealize_style = theme_unrealize_style;
  engine->destroy_rc_style = theme_destroy_rc_style;
  engine->destroy_style = theme_destroy_style;
  engine->set_background = NULL;

  gdk_rgb_init ();
}

G_MODULE_EXPORT void
theme_exit (void)
{
}

/* The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */
G_MODULE_EXPORT const gchar* g_module_check_init (GModule *module);
const gchar*
g_module_check_init (GModule *module)
{
  return gtk_check_version (GTK_MAJOR_VERSION,
			    GTK_MINOR_VERSION,
			    GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}
