/*
    $Id: info_private.c,v 1.1.2.9 2003/05/18 01:53:50 rocky Exp $

    Copyright (C) 2003 Rocky Bernstein <rocky@panix.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Foundation
    Software, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/* 
   Like vcdinfo but exposes more of the internal structure. It is probably
   better to use vcdinfo, when possible.
*/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <stdio.h>
#include <stddef.h>
#include <errno.h>

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <cdio/cdio.h>
#include <cdio/util.h>

#include <libvcd/types.h>
#include <libvcd/files.h>
#include <libvcd/iso9660_private.h>

#include <libvcd/info.h>
#include <libvcd/info_private.h>

/* Private headers */
#include "data_structures.h"
#include "pbc.h"
#include "xa.h"

static const char _rcsid[] = "$Id: info_private.c,v 1.1.2.9 2003/05/18 01:53:50 rocky Exp $";

#define BUF_COUNT 16
#define BUF_SIZE 80

/* Return a pointer to a internal free buffer */
static char *
_getbuf (void)
{
  static char _buf[BUF_COUNT][BUF_SIZE];
  static int _num = -1;
  
  _num++;
  _num %= BUF_COUNT;

  memset (_buf[_num], 0, BUF_SIZE);

  return _buf[_num];
}

const char *
vcdinf_area_str (const struct psd_area_t *_area)
{
  char *buf;

  if (!_area->x1  
      && !_area->y1
      && !_area->x2
      && !_area->y2)
    return "disabled";

  buf = _getbuf ();

  snprintf (buf, BUF_SIZE, "[%3d,%3d] - [%3d,%3d]",
            _area->x1, _area->y1,
            _area->x2, _area->y2);
            
  return buf;
}

/*! 
  Comparison routine used in sorting. We compare LIDs and if those are 
  equal, use the offset.
  Note: we assume an unassigned LID is 0 and this compares as a high value.
  
  NOTE: Consider making static.
*/
int
vcdinf_lid_t_cmp (vcdinfo_offset_t *a, vcdinfo_offset_t *b)
{
  if (a->lid && b->lid)
    {
      if (a->lid > b->lid) return +1;
      if (a->lid < b->lid) return -1;
      vcd_warn ("LID %d at offset %d has same nunber as LID of offset %d", 
                a->lid, a->offset, b->offset);
    } 
  else if (a->lid) return -1;
  else if (b->lid) return +1;

  /* Failed to sort on LID, try offset now. */

  if (a->offset > b->offset) return +1;
  if (a->offset < b->offset) return -1;
  
  /* LIDS and offsets are equal. */
  return 0;
}

/*
  This fills in unassigned LIDs in the offset table.  Due to
  "rejected" LOT entries, some of these might not have gotten filled
  in while scanning PBC (if in fact there even was a PBC).

  Note: We assume that an unassigned LID is one whose value is 0.
 */
static void
vcdinf_update_offset_list(struct _vcdinf_pbc_ctx *obj, bool extended)
{
  if (NULL==obj) return;
  {
    VcdListNode *node;
    VcdList *unused_lids = _vcd_list_new();
    VcdListNode *next_unused_node = _vcd_list_begin(unused_lids);
    
    unsigned int last_lid=0;
    VcdList *offset_list = extended ? obj->offset_x_list : obj->offset_list;
    
    lid_t max_seen_lid=0;

    _VCD_LIST_FOREACH (node, offset_list)
      {
        vcdinfo_offset_t *ofs = _vcd_list_node_data (node);
        if (!ofs->lid) {
          /* We have a customer! Assign a LID from the free pool
             or take one from the end if no skipped LIDs.
          */
          VcdListNode *node=_vcd_list_node_next(next_unused_node);
          if (node != NULL) {
            lid_t *next_unused_lid=_vcd_list_node_data(node);
            ofs->lid = *next_unused_lid;
            next_unused_node=node;
          } else {
            max_seen_lid++;
            ofs->lid = max_seen_lid;
          }
        } else {
          /* See if we've skipped any LID numbers. */
          last_lid++;
          while (last_lid != ofs->lid ) {
            lid_t * lid=_vcd_malloc (sizeof(lid_t));
            *lid = last_lid;
            _vcd_list_append(unused_lids, lid);
          }
          if (last_lid > max_seen_lid) max_seen_lid=last_lid;
        }
      }
    _vcd_list_free(unused_lids, true);
  }
}

/*!
   Calls recursive routine to populate obj->offset_list or obj->offset_x_list
   by going through LOT.
*/
void
vcdinf_visit_lot (struct _vcdinf_pbc_ctx *obj)
{
  const LotVcd *lot = obj->extended ? obj->lot_x : obj->lot;
  unsigned int n, tmp;

  if (obj->extended) {
    if (!obj->psd_x_size) return;
  } else if (!obj->psd_size) return;

  for (n = 0; n < LOT_VCD_OFFSETS; n++)
    if ((tmp = vcdinf_get_lot_offset(lot, n)) != PSD_OFS_DISABLED)
      vcdinf_visit_pbc (obj, n + 1, tmp, true);

  _vcd_list_sort (obj->extended ? obj->offset_x_list : obj->offset_list, 
                  (_vcd_list_cmp_func) vcdinf_lid_t_cmp);

  /* Now really complete the offset table with LIDs.  This routine
     might obviate the need for vcdinf_visit_pbc() or some of it which is
     more complex. */
  vcdinf_update_offset_list(obj, obj->extended);
}

/*!
   Recursive routine to populate obj->offset_list or obj->offset_x_list
   by reading playback control entries referred to via lid.
*/
void
vcdinf_visit_pbc (struct _vcdinf_pbc_ctx *obj, lid_t lid, unsigned int offset, 
                  bool in_lot)
{
  VcdListNode *node;
  vcdinfo_offset_t *ofs;
  unsigned int psd_size  = obj->extended ? obj->psd_x_size : obj->psd_size;
  const uint8_t *psd = obj->extended ? obj->psd_x : obj->psd;
  unsigned int _rofs = offset * obj->offset_mult;
  VcdList *offset_list;

  vcd_assert (psd_size % 8 == 0);

  switch (offset)
    {
    case PSD_OFS_DISABLED:
    case PSD_OFS_MULTI_DEF:
    case PSD_OFS_MULTI_DEF_NO_NUM:
      return;

    default:
      break;
    }

  if (_rofs >= psd_size)
    {
      if (obj->extended)
	vcd_error ("psd offset out of range in extended PSD"
		   " (try --no-ext-psd option)");
      else
        vcd_warn ("psd offset out of range (%d >= %d)", _rofs, psd_size);
      return;
    }

  vcd_assert (_rofs < psd_size);

  if (!obj->offset_list)
    obj->offset_list = _vcd_list_new ();

  if (!obj->offset_x_list)
    obj->offset_x_list = _vcd_list_new ();

  if (obj->extended) {
    offset_list = obj->offset_x_list;
  } else 
    offset_list = obj->offset_list;

  _VCD_LIST_FOREACH (node, offset_list)
    {
      ofs = _vcd_list_node_data (node);

      if (offset == ofs->offset)
        {
          if (in_lot)
            ofs->in_lot = true;

          if (lid) {
            /* Our caller thinks she knows what our LID is.
               This should help out getting the LID for end descriptors
               if not other things as well.
             */
            ofs->lid = lid;
          }
          
          ofs->ext = obj->extended;

          return; /* already been there... */
        }
    }

  ofs = _vcd_malloc (sizeof (vcdinfo_offset_t));

  ofs->ext    = obj->extended;
  ofs->in_lot = in_lot;
  ofs->lid    = lid;
  ofs->offset = offset;
  ofs->type   = psd[_rofs];

  switch (ofs->type)
    {
    case PSD_TYPE_PLAY_LIST:
      _vcd_list_append (offset_list, ofs);
      {
        const PsdPlayListDescriptor *d = (const void *) (psd + _rofs);
        const lid_t lid = vcdinf_get_lid_from_pld(d);

        if (!ofs->lid)
          ofs->lid = lid;
        else 
          if (ofs->lid != lid)
            vcd_warn ("LOT entry assigned LID %d, but descriptor has LID %d",
                      ofs->lid, lid);

        vcdinf_visit_pbc (obj, 0, vcdinf_get_prev_from_pld(d), false);
        vcdinf_visit_pbc (obj, 0, vcdinf_get_next_from_pld(d), false);
        vcdinf_visit_pbc (obj, 0, vcdinf_get_return_from_pld(d), false);
      }
      break;

    case PSD_TYPE_EXT_SELECTION_LIST:
    case PSD_TYPE_SELECTION_LIST:
      _vcd_list_append (offset_list, ofs);
      {
        const PsdSelectionListDescriptor *d =
          (const void *) (psd + _rofs);

        int idx;

        if (!ofs->lid)
          ofs->lid = uint16_from_be (d->lid) & 0x7fff;
        else 
          if (ofs->lid != (uint16_from_be (d->lid) & 0x7fff))
            vcd_warn ("LOT entry assigned LID %d, but descriptor has LID %d",
                      ofs->lid, uint16_from_be (d->lid) & 0x7fff);

        vcdinf_visit_pbc (obj, 0, vcdinf_get_prev_from_psd(d), false);
        vcdinf_visit_pbc (obj, 0, vcdinf_get_next_from_psd(d), false);
        vcdinf_visit_pbc (obj, 0, vcdinf_get_return_from_psd(d), false);
        vcdinf_visit_pbc (obj, 0, vcdinf_get_default_from_psd(d), false);
        vcdinf_visit_pbc (obj, 0, uint16_from_be (d->timeout_ofs), false);

        for (idx = 0; idx < vcdinf_get_num_selections(d); idx++)
          vcdinf_visit_pbc (obj, 0, vcdinf_get_offset_from_psd(d, idx), false);
        
      }
      break;

    case PSD_TYPE_END_LIST:
      _vcd_list_append (offset_list, ofs);
      break;

    default:
      vcd_warn ("corrupt PSD???????");
      free (ofs);
      return;
      break;
    }
}

/*!
   Return a string containing the VCD album id, or NULL if there is 
   some problem in getting this. 
*/
const char *
vcdinf_get_album_id(const InfoVcd *info)
{
  if (NULL==info) return NULL;
  return vcdinfo_strip_trail (info->album_desc, MAX_ALBUM_LEN);
}

/*!
  Return the VCD application ID.
  NULL is returned if there is some problem in getting this. 
*/
const char * 
vcdinf_get_application_id(const pvd_t *pvd)
{
  if (NULL==pvd) return NULL;
  return(vcdinfo_strip_trail(pvd->application_id, MAX_APPLICATION_ID));
}

/*!
  Return the base selection number. VCD_INVALID_BSN is returned if there
  is an error.
*/
unsigned int
vcdinf_get_bsn(const PsdSelectionListDescriptor *psd)
{
  if (NULL==psd) return VCDINFO_INVALID_BSN;
  return(psd->bsn);
}

/**
 * \fn vcdinfo_get_default_from_psd(const PsdSelectionListDescriptor *psd);
 * \brief Get next offset for a given PSD selector descriptor. 
 * \return VCDINFO_INVALID_OFFSET is returned on error or if psd is
 * NULL. Otherwise the LID offset is returned.
 */
lid_t
vcdinf_get_default_from_psd(const PsdSelectionListDescriptor *psd)
{
  if (NULL == psd) return VCDINFO_INVALID_OFFSET;
  return uint16_from_be (psd->default_ofs);
}

/*!  Return the starting LBA (logical block address) for sequence
  entry_num in obj.  VCDINFO_NULL_LBA is returned if there is no entry.
*/
lba_t
vcdinf_get_entry_lba(const EntriesVcd *entries, unsigned int entry_num)
{
  const msf_t *msf = vcdinf_get_entry_msf(entries, entry_num);
  return (msf != NULL) ? cdio_msf_to_lba(msf) : VCDINFO_NULL_LBA;
}

/*!  Return the starting MSF (minutes/secs/frames) for sequence
  entry_num in obj.  NULL is returned if there is no entry.
  The first entry number is 0.
*/
const msf_t *
vcdinf_get_entry_msf(const EntriesVcd *entries, unsigned int entry_num)
{
  const unsigned int entry_count = uint16_from_be (entries->entry_count);
  return entry_num < entry_count ?
    &(entries->entry[entry_num].msf)
    : NULL;
}

/*!
   Return a string giving VCD format (VCD 1.0 VCD 1.1, SVCD, ... 
   for this object.
*/
const char *
vcdinf_get_format_version_str (vcd_type_t vcd_type) 
{
  switch (vcd_type)
    {
    case VCD_TYPE_VCD:
      return ("VCD 1.0");
      break;
    case VCD_TYPE_VCD11:
      return ("VCD 1.1");
      break;
    case VCD_TYPE_VCD2:
      return ("VCD 2.0");
      break;
    case VCD_TYPE_SVCD:
      return ("SVCD");
      break;
    case VCD_TYPE_HQVCD:
      return ("HQVCD");
      break;
    case VCD_TYPE_INVALID:
      return ("INVALID");
      break;
    default:
      return ( "????");
    }
}

/*!
  Get the item id for a given selection-list descriptor. 
  VCDINFO_REJECTED_MASK is returned on error or if psd is NULL. 
*/
uint16_t
vcdinf_get_itemid_from_psd(const PsdSelectionListDescriptor *psd)
{
  return (psd != NULL) ? uint16_from_be(psd->itemid) : VCDINFO_REJECTED_MASK;
}

/* Get the LID from a given play-list descriptor. 
   VCDINFO_REJECTED_MASK is returned d on error or pld is NULL. 
*/
lid_t
vcdinf_get_lid_from_pld(const PsdPlayListDescriptor *pld)
{
  return (pld != NULL) 
    ? uint16_from_be (pld->lid) & VCDINFO_LID_MASK
    : VCDINFO_REJECTED_MASK;
}

/*!
  Get the LID from a given selection-list descriptor. 
  VCDINFO_REJECTED_MASK is returned on error or psd is NULL. 
*/
lid_t
vcdinf_get_lid_from_psd(const PsdSelectionListDescriptor *psd)
{
  return (psd != NULL) 
    ? uint16_from_be (psd->lid) & VCDINFO_LID_MASK
    : VCDINFO_REJECTED_MASK;
}

/*!
  Get the LID rejected status for a given PSD selector descriptor. 
  true is also returned d is NULL. 
*/
bool
vcdinf_get_lid_rejected_from_psd(const PsdSelectionListDescriptor *psd)
{
  return (psd != NULL) 
    ? vcdinfo_is_rejected(uint16_from_be(psd->lid)) 
    : true;
}

/*!
  Return LOT offset
*/
uint16_t
vcdinf_get_lot_offset (const LotVcd *lot, unsigned int n) 
{
  return uint16_from_be (lot->offset[n]);
}

/*!
  Return loop count. 0 is infinite loop.
*/
uint16_t
vcdinf_get_loop_count (const PsdSelectionListDescriptor *psd) 
{
  return 0x7f & psd->loop;
}

/**
 \fn vcdinfo_get_next_from_pld(const PsdPlayListDescriptor *pld); 
 \brief  Get next offset for a given PSD selector descriptor.  
 \return  VCDINFO_INVALID_OFFSET is returned on error or if pld has no "next"
 entry or pld is NULL. Otherwise the LID offset is returned.
 */
lid_t
vcdinf_get_next_from_pld(const PsdPlayListDescriptor *pld)
{
  if (NULL == pld) return VCDINFO_INVALID_OFFSET;
  return uint16_from_be (pld->next_ofs);
}

/**
 * \fn vcdinfo_get_next_from_psd(const PsdSelectionListDescriptor *psd);
 * \brief Get next offset for a given PSD selector descriptor. 
 * \return VCDINFO_INVALID_OFFSET is returned on error or if psd is
 * NULL. Otherwise the LID offset is returned.
 */
lid_t
vcdinf_get_next_from_psd(const PsdSelectionListDescriptor *psd)
{
  if (NULL == psd) return VCDINFO_INVALID_OFFSET;
  return uint16_from_be (psd->next_ofs);
}

/*!
  Return the number of entries in the VCD.
*/
unsigned int
vcdinf_get_num_entries(const EntriesVcd *entries)
{
  if (NULL==entries) return 0;
  return (uint16_from_be (entries->entry_count));
}

/*!
  Return the number of menu selections for selection list descriptor psd.
*/
unsigned int
vcdinf_get_num_selections(const PsdSelectionListDescriptor *psd)
{
  return psd->nos;
}

/*!
  Return the number of segments in the VCD. 
*/
unsigned int
vcdinf_get_num_segments(const InfoVcd *info)
{
  if (NULL==info) return 0;
  return (uint16_from_be (info->item_count));
}

/*!
  Return number of LIDs. 
*/
lid_t
vcdinf_get_num_LIDs (const InfoVcd *info) 
{
  if (NULL==info) return 0;
  /* Should probably use _vcd_pbc_max_lid instead? */
  return uint16_from_be (info->lot_entries);
}

/*!
  \fn vcdinfo_get_offset_from_lid(const vcdinfo_obj_t *obj, unsigned int entry_num);
  \brief Get offset entry_num for a given LID. 
  \return VCDINFO_INVALID_OFFSET is returned if obj on error or obj
  is NULL. Otherwise the LID offset is returned.
*/
uint16_t
vcdinf_get_offset_from_lid(const vcdinfo_obj_t *obj, lid_t lid,
                           unsigned int entry_num) 
{
  PsdListDescriptor pxd;

  if (obj == NULL) return VCDINFO_INVALID_OFFSET;
  vcdinf_get_pxd_from_lid(obj, &pxd, lid);

  switch (pxd.descriptor_type) {
  case PSD_TYPE_SELECTION_LIST:
  case PSD_TYPE_EXT_SELECTION_LIST:
    if (pxd.psd == NULL) return VCDINFO_INVALID_OFFSET;
    return vcdinf_get_offset_from_psd(pxd.psd, entry_num-1);
    break;
  case PSD_TYPE_PLAY_LIST:
    /* FIXME: There is an array of items */
  case PSD_TYPE_END_LIST:
  case PSD_TYPE_COMMAND_LIST:
    return VCDINFO_INVALID_OFFSET;
  }
  return VCDINFO_INVALID_OFFSET;

}

/**
 * \fn vcdinfo_get_offset_from_psd(const PsdSelectionListDescriptor *d, unsigned int entry_num);
 * \brief Get offset entry_num for a given PSD selector descriptor. 
 * \return VCDINFO_INVALID_OFFSET is returned if d on error or d is
 * NULL. Otherwise the LID offset is returned.
 */
uint16_t
vcdinf_get_offset_from_psd(const PsdSelectionListDescriptor *psd, 
                           unsigned int entry_num) 
{
  return (psd != NULL && entry_num < vcdinf_get_num_selections(psd))
    ? uint16_from_be (psd->ofs[entry_num]) : VCDINFO_INVALID_OFFSET;
}

/*!
  Return the playlist item i in d. 
*/
uint16_t
vcdinf_get_play_item_from_pld(const PsdPlayListDescriptor *pld, unsigned int i)
{
  if (NULL==pld) return 0;
  return  uint16_from_be(pld->itemid[i]);
}

/*!
  Get play time value for PsdPlayListDescriptor *d.
  Time is in 1/15-second units.
*/
uint16_t
vcdinf_get_play_time (const PsdPlayListDescriptor *d) 
{
  if (NULL==d) return 0;
  return uint16_from_be (d->ptime);
}

/*!
   Return a string containing the VCD preparer id with trailing
   blanks removed.
*/
const char *
vcdinf_get_preparer_id(const pvd_t *pvd)
{
  if (NULL==pvd) return NULL;
  return(vcdinfo_strip_trail(pvd->preparer_id, MAX_PREPARER_ID));
}

/**
 \fn vcdinf_get_prev_from_pld(const PsdPlayListDescriptor *pld);
 \brief Get prev offset for a given PSD selector descriptor. 
 \return  VCDINFO_INVALID_OFFSET is returned on error or if pld has no "prev"
 entry or pld is NULL. Otherwise the LID offset is returned.
 */
lid_t
vcdinf_get_prev_from_pld(const PsdPlayListDescriptor *pld)
{
  return (pld != NULL) ? 
    uint16_from_be (pld->prev_ofs) : VCDINFO_INVALID_OFFSET;
}

/**
 \fn vcdinf_get_prev_from_psd(const PsdSelectionListDescriptor *psd);
 \brief Get prev offset for a given PSD selector descriptor. 
 \return  VCDINFO_INVALID_OFFSET is returned on error or if psd has no "prev"
 entry or psd is NULL. Otherwise the LID offset is returned.
 */
lid_t
vcdinf_get_prev_from_psd(const PsdSelectionListDescriptor *psd)
{
  return (psd != NULL) ? 
    uint16_from_be (psd->prev_ofs) : VCDINFO_INVALID_OFFSET;
}

/*!
  Return number of bytes in PSD. 
*/
uint32_t
vcdinf_get_psd_size (const InfoVcd *info)
{
  if (NULL==info) return 0;
  return uint32_from_be (info->psd_size);
}

/*!
   Return a string containing the VCD publisher id with trailing
   blanks removed.
*/
const char *
vcdinf_get_publisher_id(const pvd_t *pvd)
{
  if (NULL==pvd) return NULL;
  return(vcdinfo_strip_trail(pvd->publisher_id, MAX_PUBLISHER_ID));
}

/*!
  Get the PSD Selection List Descriptor for a given lid.
  NULL is returned if error or not found.
*/
static bool
_vcdinf_get_pxd_from_lid(const vcdinfo_obj_t *obj, PsdListDescriptor *pxd,
                         uint16_t lid, bool ext) 
{
  VcdListNode *node;
  unsigned mult = obj->info.offset_mult;
  const uint8_t *psd = ext ? obj->psd_x : obj->psd;
  VcdList *offset_list = ext ? obj->offset_x_list : obj->offset_list;

  if (offset_list == NULL) return false;
  
  _VCD_LIST_FOREACH (node, offset_list)
    {
      vcdinfo_offset_t *ofs = _vcd_list_node_data (node);
      unsigned _rofs = ofs->offset * mult;

      pxd->descriptor_type = psd[_rofs];

      switch (pxd->descriptor_type)
        {
        case PSD_TYPE_PLAY_LIST:
          {
            pxd->pld = (PsdPlayListDescriptor *) (psd + _rofs);
            if (vcdinf_get_lid_from_pld(pxd->pld) == lid) {
              return true;
            }
            break;
          }

        case PSD_TYPE_EXT_SELECTION_LIST:
        case PSD_TYPE_SELECTION_LIST: 
          {
            pxd->psd = (PsdSelectionListDescriptor *) (psd + _rofs);
            if (vcdinf_get_lid_from_psd(pxd->psd) == lid) {
              return true;
            }
            break;
          }
        default: ;
        }
    }
  return false;
}

/*!
  Get the PSD Selection List Descriptor for a given lid.
  False is returned if not found.
*/
bool
vcdinf_get_pxd_from_lid(const vcdinfo_obj_t *obj, PsdListDescriptor *pxd,
                         uint16_t lid)
{
  if (_vcdinf_get_pxd_from_lid(obj, pxd, lid, true))
    return true;
  return _vcdinf_get_pxd_from_lid(obj, pxd, lid, false);
}

/**
 \fn vcdinf_get_return_from_pld(const PsdPlayListDescriptor *pld);
 \brief Get return offset for a given PLD selector descriptor. 
 \return  VCDINFO_INVALID_OFFSET is returned on error or if pld has no 
 "return" entry or pld is NULL. Otherwise the LID offset is returned.
 */
lid_t
vcdinf_get_return_from_pld(const PsdPlayListDescriptor *pld)
{
  return (pld != NULL) ? 
    uint16_from_be (pld->return_ofs) : VCDINFO_INVALID_OFFSET;
}

/**
 * \fn vcdinfo_get_return_from_psd(const PsdSelectionListDescriptor *psd);
 * \brief Get return offset for a given PSD selector descriptor. 
 \return  VCDINFO_INVALID_OFFSET is returned on error or if psd has no 
 "return" entry or psd is NULL. Otherwise the LID offset is returned.
 */
uint16_t
vcdinf_get_return_from_psd(const PsdSelectionListDescriptor *psd)
{
  return (psd != NULL) ? 
    uint16_from_be (psd->return_ofs) : VCDINFO_INVALID_OFFSET;
}

/*!
   Return a string containing the VCD system id with trailing
   blanks removed.
*/
const char *
vcdinf_get_system_id(const pvd_t *pvd)
{
  if (NULL==pvd) return NULL;
  return(vcdinfo_strip_trail(pvd->system_id, MAX_SYSTEM_ID));
}

/*!
  Return the track number for entry n in obj. The first track starts
  at 1. Note this is one less than the track number reported in vcddump.
  (We don't count the header track?)
*/
track_t
vcdinf_get_track(const EntriesVcd *entries, const unsigned int entry_num)
{
  const unsigned int entry_count = uint16_from_be (entries->entry_count);
  /* Note entry_num is 0 origin. */
  return entry_num < entry_count ?
    from_bcd8 (entries->entry[entry_num].n):
    VCDINFO_INVALID_TRACK;
}

/*!
  Return the VCD volume count - the number of CD's in the collection.
*/
unsigned int 
vcdinf_get_volume_count(const InfoVcd *info) 
{
  if (NULL==info) return 0;
  return(uint16_from_be( info->vol_count));
}

/*!
  Return the VCD ID.
*/
const char *
vcdinf_get_volume_id(const pvd_t *pvd) 
{
  if (NULL == pvd) return NULL;
  return(vcdinfo_strip_trail(pvd->volume_id, MAX_VOLUME_ID));
}

/*!
  Return the VCD volumeset ID.
  NULL is returned if there is some problem in getting this. 
*/
const char *
vcdinf_get_volumeset_id(const pvd_t *pvd)
{
  if ( NULL == pvd ) return NULL;
  return vcdinfo_strip_trail(pvd->volume_set_id, MAX_VOLUMESET_ID);
}

/*!
  Return the VCD volume num - the number of the CD in the collection.
  This is a number between 1 and the volume count.
*/
unsigned int
vcdinf_get_volume_num(const InfoVcd *info)
{
  if (NULL == info) return 0;
  return uint16_from_be(info->vol_id);
}

int
vcdinf_get_wait_time (uint16_t wtime)
{
  /* Note: this doesn't agree exactly with _wtime */
  if (wtime < 61)
    return wtime;
  else if (wtime < 255)
    return (wtime - 60) * 10 + 60;
  else
    return -1;
}

/*!
  Return true if loop has a jump delay
*/
bool
vcdinf_has_jump_delay (const PsdSelectionListDescriptor *psd) 
{
  if (NULL==psd) return false;
  return ((0x80 & psd->loop) != 0);
}
  
/*!
  Set up img structure for reading from a particular medium. 
 
   source_name is the device or file to use for inspection, and
   source_type indicates if this is a device or a file.
   access_mode gives special access options for reading.
    
   True is returned if everything went okay; 
 */

bool
vcdinf_open(/*out*/ CdIo **img, char *source_name[],  
            vcdinfo_source_t source_type, const char access_mode[]) 
{
  char is_sector_2352 = true;
  bool null_source = NULL == *source_name;
  
  if (VCDINFO_SOURCE_AUTO == source_type) {
    if (null_source) 
      source_type = VCDINFO_SOURCE_DEVICE;
    else {
      struct stat buf;
      if (0 != stat(*source_name, &buf)) {
        vcd_error ("Can't stat file %s:", strerror(errno));
        return false;
      }
      if (S_ISBLK(buf.st_mode) || S_ISCHR(buf.st_mode)) {
        source_type = VCDINFO_SOURCE_DEVICE;
      } else if (S_ISREG(buf.st_mode)) {
        /* FIXME: check to see if is a text file. If so, then 
           set VCDINFO_SOURCE_CUE. */
        int i=strlen(*source_name)-strlen("bin");
        if (i > 0
            && ( (*source_name)[i]   =='n' || (*source_name)[i]   =='N' )
            && ( (*source_name)[i+1] =='r' || (*source_name)[i+1] =='R' )
            && ( (*source_name)[i+2] =='g' || (*source_name)[i+2] =='G' ) )
          source_type = VCDINFO_SOURCE_NRG;
        else {
          char *bin_file = cdio_is_cuefile(*source_name);
          if (NULL != bin_file) {
            source_type = VCDINFO_SOURCE_CUE;
            free(bin_file);
          } else
            source_type = VCDINFO_SOURCE_BIN;
        } 
      } else {
        vcd_error ("Source file `%s' should either be a block device "
                   "or a regular file", *source_name);
        return false;
      }
    }
  }
  
  switch (source_type) {
  case VCDINFO_SOURCE_DEVICE:
    *img = cdio_open_cd(*source_name);
    break;
  case VCDINFO_SOURCE_SECTOR_2336:
    is_sector_2352 = false;
    /* Fall through to next case */
  case VCDINFO_SOURCE_BIN: 
    {
      *img = cdio_open_bincue(*source_name);
      if (NULL != *img) {
        const char *cue_file = cdio_get_arg(*img, "cue");
        if (NULL == cue_file) {
          vcd_error ("Can't find corresponding CUE or TOC file for `%s'", 
                     *source_name);
          return false;
        }
      }
      break;
    }
  case VCDINFO_SOURCE_CUE:
    *img = cdio_open_cue(*source_name);
    break;
  case VCDINFO_SOURCE_NRG:
    *img = cdio_open_nrg(*source_name);
    break;
  default: 
    return false;
  }

  if (NULL == *img) {
    vcd_error ("Error opening`%s'", *source_name);
    return false;
  }
  
  if (null_source) {
    *source_name=cdio_get_default_device(*img);
  }

  if (access_mode != NULL) 
    cdio_set_arg (*img, "access-mode", access_mode);

  return NULL != *source_name;

}



/* 
 * Local variables:
 *  c-file-style: "gnu"
 *  tab-width: 8
 *  indent-tabs-mode: nil
 * End:
 */
