/* X-Chat
 * Copyright (C) 1998 Peter Zelezny.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "xchat.h"

#include "cfgfiles.h"
#include "fe.h"
#include "server.h"
#include "text.h"
#include "xchatc.h"

#include "servlist.h"



GSList *network_list = 0;


void
servlist_connect (session *sess, ircnet *net)
{
	ircserver *ircserv;
	GSList *list;
	char *port;
	server *serv;

	if (!sess)
		sess = new_ircwindow (NULL, NULL, SESS_SERVER);

	serv = sess->server;

	/* connect to the currently selected Server-row */
	list = g_slist_nth (net->servlist, net->selected);
	if (!list)
		list = net->servlist;
	ircserv = list->data;

	/* incase a protocol switch is added to the servlist gui */
	server_fill_her_up (sess->server);

	if (net->autojoin)
		safe_strcpy (sess->willjoinchannel, net->autojoin,
						 sizeof (sess->willjoinchannel));
	if (net->pass)
		safe_strcpy (serv->password, net->pass, sizeof (serv->password));

	if (!(net->flags & FLAG_USE_GLOBAL) && net->nick)
		strcpy (serv->nick, net->nick);
	else
		strcpy (serv->nick, prefs.nick1);

	serv->dont_use_proxy = (net->flags & FLAG_USE_PROXY) ? FALSE : TRUE;

#ifdef USE_OPENSSL
	serv->use_ssl = (net->flags & FLAG_USE_SSL) ? TRUE : FALSE;
	serv->accept_invalid_cert =
		(net->flags & FLAG_ALLOW_INVALID) ? TRUE : FALSE;
#endif

	if (net->flags & FLAG_CYCLE)
		/* needed later to do cycling */
		serv->network = net;
	else
		serv->network = NULL;

	port = strrchr (ircserv->hostname, '/');
	if (port)
	{
		*port = 0;
		serv->connect (serv, ircserv->hostname, atoi (port + 1), FALSE);
		*port = '/';
	} else
		serv->connect (serv, ircserv->hostname, 6667, FALSE);

	if (net->command)
	{
		if (serv->eom_cmd)
			free (serv->eom_cmd);
		serv->eom_cmd = strdup (net->command);
	}
}

int
servlist_auto_connect (session *sess)
{
	GSList *list = network_list;
	ircnet *net;
	int ret = 0;

	while (list)
	{
		net = list->data;

		if (net->flags & FLAG_AUTO_CONNECT)
		{
			servlist_connect (sess, net);
			ret = 1;
		}

		list = list->next;
	}

	return ret;
}

static gint
servlist_cycle_cb (server *serv)
{
	servlist_connect (serv->server_session, serv->network);
	PrintTextf (serv->server_session,
		"Cycling to next server in %s...\n", ((ircnet *)serv->network)->name);

	return 0;
}

int
servlist_cycle (server *serv)
{
	ircnet *net;
	int max, del;

	net = serv->network;
	if (net)
	{
		max = g_slist_length (net->servlist);
		if (max > 1)
		{
			net->selected++;
			if (net->selected == max)
				net->selected = 0;

			del = prefs.recon_delay * 1000;
			if (del < 1000)
				del = 500;				  /* so it doesn't block the gui */

			if (del)
				serv->recondelay_tag = fe_timeout_add (del, servlist_cycle_cb, serv);
			else
				servlist_connect (serv->server_session, net);

			return TRUE;
		}
	}

	return FALSE;
}

ircserver *
servlist_server_find (ircnet *net, char *name, int *pos)
{
	GSList *list = net->servlist;
	ircserver *serv;
	int i = 0;

	while (list)
	{
		serv = list->data;
		if (strcmp (serv->hostname, name) == 0)
		{
			if (pos)
				*pos = i;
			return serv;
		}
		i++;
		list = list->next;
	}

	return NULL;
}

ircnet *
servlist_net_find (char *name)
{
	GSList *list = network_list;
	ircnet *net;

	while (list)
	{
		net = list->data;
		if (strcmp (net->name, name) == 0)
			return net;
		list = list->next;
	}

	return NULL;
}

void
servlist_server_add (ircnet *net, char *name)
{
	ircserver *serv;

	serv = malloc (sizeof (ircserver));
	memset (serv, 0, sizeof (ircserver));
	serv->hostname = strdup (name);

	net->servlist = g_slist_append (net->servlist, serv);
}

void
servlist_server_remove (ircnet *net, ircserver *serv)
{
	free (serv->hostname);
	free (serv);
	net->servlist = g_slist_remove (net->servlist, serv);
}

void
servlist_server_remove_all (ircnet *net)
{
	ircserver *serv;

	while (net->servlist)
	{
		serv = net->servlist->data;
		servlist_server_remove (net, serv);
	}
}

void
servlist_net_remove (ircnet *net)
{
	GSList *list;
	server *serv;

	servlist_server_remove_all (net);
	network_list = g_slist_remove (network_list, net);
	free (net->name);
	free (net);

	/* for safety */
	list = serv_list;
	while (list)
	{
		serv = list->data;
		if (serv->network == net)
			serv->network = NULL;
		list = list->next;
	}
}

ircnet *
servlist_net_add (char *name, char *comment)
{
	ircnet *net;

	net = malloc (sizeof (ircnet));
	memset (net, 0, sizeof (ircnet));
	net->name = strdup (name);
//	net->comment = strdup (comment);
	net->flags = FLAG_CYCLE | FLAG_USE_GLOBAL | FLAG_USE_PROXY;

	network_list = g_slist_append (network_list, net);

	return net;
}

struct defaultserver
{
	char *network;
	char *host;
};

static const struct defaultserver def[] =
{
	{"AustNet",		""},
	{0,				"us.austnet.org"},
	{0,				"ca.austnet.org"},
	{0,				"au.austnet.org"},

	{"ChatJunkies",""},
	{0,				"irc.chatjunkies.org"},

	{"DALNet",		""},
	{0,				"irc.dal.net"},

	{"EFNet",		""},
	{0,				"irc.efnet.org"},
	{0,				"irc.efnet.pl"},
	{0,				"efnet.demon.co.uk"},
	{0,				"irc.flamed.net"},
	{0,				"irc.concentric.net"},
	{0,				"irc.Qeast.net"},
	{0,				"irc.daxnet.no"},
	{0,				"irc.Prison.NET"},

	{"FreeNode",	""},
	{0,				"irc.freenode.net"},
	{0,				"irc.au.freenode.net"},
	{0,				"irc.us.freenode.net"},

	{"GalaxyNet",	""},
	{0,				"irc.galaxynet.org"},

	{"GamesNET",	""},
	{0,				"irc.GamesNET.net"},

	{"GimpNet",		""},
	{0,				"irc.gimp.org"},
	{0,				"irc.au.gimp.org"},

	{"IRCNet",		""},
	{0,				"irc.stealth.net/5550"},
	{0,				"irc.bt.net"},
	{0,				"irc.funet.fi"},
	{0,				"ircnet.demon.co.uk"},
	{0,				"ircnet.hinet.hr"},
	{0,				"irc.datacomm.ch"},
	{0,				"ircnet.kaptech.fr"},
	{0,				"irc.flashnet.it"},
	{0,				"irc.cwitaly.it"},
	{0,				"ircnet.easynet.co.uk"},

	{"QuakeNet",	""},
	{0,				"irc.quakenet.org"},

	{"OFTC",			""},
	{0,				"irc.oftc.net"},

	{"OzNet",		""},
	{0,				"sydney.oz.org"},
	{0,				"melbourne.oz.org"},

	{"Undernet",	""},
	{0,				"irc.undernet.org"},
	{0,				"us.undernet.org"},
	{0,				"eu.undernet.org"},

	{"WebChat",		""},
	{0,				"irc.webchat.org"},

	{0}
};

static void
servlist_load_defaults (void)
{
	int i = 0;
	ircnet *net = NULL;

	while (def[i].host)
	{
		if (def[i].network)
			net = servlist_net_add (def[i].network, def[i].host);
		else
			servlist_server_add (net, def[i].host);
		i++;
	}
}

void
servlist_init (void)
{
	if (!network_list)
		if (!servlist_load ())
			servlist_load_defaults ();
}

void
servlist_save (void)
{
	FILE *fp;
	char buf[256];
	ircnet *net;
	ircserver *serv;
	GSList *list;
	GSList *hlist;

	snprintf (buf, sizeof (buf), "%s/servlist_.conf", get_xdir ());
	printf("saving %s\n", buf);
	fp = fopen (buf, "w");
	if (!fp)
		return;

	fprintf (fp, "v="VERSION"\n\n");

	list = network_list;
	while (list)
	{
		net = list->data;

		fprintf (fp, "N=%s\n", net->name);
		if (net->nick)
			fprintf (fp, "I=%s\n", net->nick);
		if (net->user)
			fprintf (fp, "U=%s\n", net->user);
		if (net->real)
			fprintf (fp, "R=%s\n", net->real);
		if (net->pass)
			fprintf (fp, "P=%s\n", net->pass);
		if (net->autojoin)
			fprintf (fp, "J=%s\n", net->autojoin);
		if (net->command)
			fprintf (fp, "C=%s\n", net->command);
		fprintf (fp, "F=%d\nD=%d\n", net->flags, net->selected);

		hlist = net->servlist;
		while (hlist)
		{
			serv = hlist->data;
			fprintf (fp, "S=%s\n", serv->hostname);
			hlist = hlist->next;
		}
		fprintf (fp, "\n");

		list = list->next;
	}

	fclose (fp);
}

int
servlist_load (void)
{
	FILE *fp;
	char buf[256];
	int len;
	ircnet *net = NULL;

	snprintf (buf, sizeof (buf), "%s/servlist_.conf", get_xdir ());
	fp = fopen (buf, "r");
	if (!fp)
		return FALSE;

	while (fgets (buf, sizeof (buf) - 1, fp))
	{
		len = strlen (buf);
		buf[len-1] = 0;	/* remove the trailing \n */
		switch (buf[0])
		{
			case 'N':
				net = servlist_net_add (buf + 2, /* comment */ "");
				break;
			case 'I':
				net->nick = strdup (buf + 2);
				break;
			case 'U':
				net->user = strdup (buf + 2);
				break;
			case 'R':
				net->real = strdup (buf + 2);
				break;
			case 'P':
				net->pass = strdup (buf + 2);
				break;
			case 'J':
				net->autojoin = strdup (buf + 2);
				break;
			case 'C':
				net->command = strdup (buf + 2);
				break;
			case 'F':
				net->flags = atoi (buf + 2);
				break;
			case 'D':
				net->selected = atoi (buf + 2);
				break;
			case 'S':	/* new server/hostname for this network */
				servlist_server_add (net, buf + 2);
				break;
		}
	}
	fclose (fp);

	return TRUE;
}
