/*
 * Copyright (C) 2000-2003 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * Compact Disc Digital Audio (CDDA) Input Plugin 
 *   by Mike Melanson (melanson@pcisys.net)
 * 
 * Completely rewritten and more fully completed
 *   by R. Bernstein (rocky@panix.com)
 *
 * to use *LIBRARIES* 
 *    libcdio - for disc reading and CD control
 *    libcddb - for CDDB processing 
 *
 * If you find yourself adding OS-specific code here, you are probably
 * doing something wrong.
 *
 * If you find yourself adding lots of xine-independent code here,
 * consider sharing your considerable coding talents with other
 * projects that also might use this code in a way other than cut-and-paste.
 * 
 * If you find the same code repeated over and over again, again there's 
 * probably better way.
 * 
 *
 * $Id: xineplug_inp_cd.c,v 1.16 2003/05/17 08:41:16 rockyb Exp $
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <errno.h>
#include <netdb.h>
#include <signal.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include <xine/xine_internal.h>
#include <xine/input_plugin.h>
#include <xine/xineutils.h>
#include <xine-extra.h>

#include <cdio/cdio.h>
#include <cdio/logging.h>

#ifdef HAVE_CDDB
#include <cddb/cddb.h>
#endif


#define SHORT_PLUGIN_NAME "CDX"
#define MRL_PREFIX "cdda:"
#define MRL_PREFIX_LEN strlen(MRL_PREFIX)

#define MRL_FORMAT(target, device, track) \
  sprintf(target, "%s/%s:%u", MRL_PREFIX, device, track);	\
  dbg_print((INPUT_DBG_MRL), "track_mrl: %s\n", target);

#define CDDB_SERVER             "freedb.freedb.org"
#define CDDB_PORT               8880

/* Debugging masks */

#define INPUT_DBG_META        1 /* Meta information */
#define INPUT_DBG_EVENT       2
#define INPUT_DBG_MRL         4 
#define INPUT_DBG_XINECALL    8 /* just xine calls to plugin */
#define INPUT_DBG_CALL       16 /* routine calls */
#define INPUT_DBG_LBA        32 /* LBA changes */
#define INPUT_DBG_INFO       64 
#define INPUT_DBG_CDIO      128
#define INPUT_DBG_CDDB      256


#define INPUT_DEBUG 1
#if INPUT_DEBUG
#define dbg_print(mask, s, args...) \
   if (cdda_debug & mask) \
     fprintf(stderr, "%s: "s, __func__ , ##args)
#else
#define dbg_print(mask, s, args...) 
#endif

static unsigned long int cdda_debug = 0;

/* Treat unit of play as the disk (with tracks) or a single track and
   stop? 
*/
typedef enum {
  track_mode,
  disk_mode,
} disk_mode_t;
  
static const char *disk_modes[] = { "track", "disk", NULL };

typedef struct 
{
  disk_mode_t  disk_mode;            /* unit of play a disk or a track? */
#ifdef HAVE_CDDB
  char         *cddb_title_format;   /* Format string of GUI display 
					title when CDDB enabled. */
  char         *cddb_comment_format; /* Format string of stream comment 
					meta when CDDB enabled.*/
#endif
  char         *title_format;        /* Format string of GUI display 
					title when CDDB not enabled. */
  char         *comment_format;      /* Format string of stream comment 
					meta when CDDB not enabled.*/
} cdda_config_t;

    
/**************************************************************************
 * xine interface functions
 *************************************************************************/

typedef struct cdda_input_class_tag cdda_input_class_t;

typedef struct {
  input_plugin_t       input_plugin;

  xine_stream_t       *stream;
  xine_event_queue_t  *event_queue;

  cdda_input_class_t  *class;        /* Class this plugin is part of.      */
  bool                 jumped;       /* True if we changed tracks or any
					sort of discontinuity in playing   */

  cdda_config_t      v_config;       /* Config stuff initially inherited   */
                                     /* from (copied) from parent          */
  track_t            cur_track;      /* The track we are currently playing */
  track_t            num_tracks;     /* The number of tracks on this CD    */
  track_t            first_track;    /* The number of first track of CD.
					This is usually 1. 	           */
  char              *mrl;            /* The MRL for the current play item  */
  lsn_t              first_frame;    /* LSN of first frame of this track   */
  lsn_t              current_frame;  /* LSN of current frame to read       */
  lsn_t              last_frame;     /* LSN of last frame of this track    */
  lsn_t              last_disk_frame;/* LSN of last frame on disk          */
  unsigned int       disk_size;      /* Size in bytes of disk              */

#ifdef HAVE_CDDB
  struct  {
    bool             have_info;      /* True if we have any info */
    cddb_disc_t     *disc;           /* libcdio uses this to get disc info */
    char            *cdiscid;
    char            *disc_title;
    char             disc_year[5];   /* Year. Probably 19XX or 20XX */
    char            *disc_artist;
    char            *disc_genre;
    cddb_cat_t       disc_category;  /* CDDB category */
    int              disc_seconds;   /* Length in seconds listed in CDDB
				        catalog. May or may not match
					length below.
				     */

    int              disc_length;    /* Length in frames of cd. Used in 
					CDDB lookups */
    unsigned int     disc_id;        /* This along with the length and 
					num_tracks below is what CDDB uses
					to look up CD info */
#if CDDB_HAD_X_CONFIGURABLE_CACHEDIR  
    char            *cachedir;
#endif
  } cddb;
#endif

} cdda_input_plugin_t;

struct cdda_input_class_tag {

  /* The below stuff most plugins will probably use. */
  input_class_t        input_class;
  xine_t              *xine;
  config_values_t     *x_conf;
  cdda_input_plugin_t *ip;                       

  cdda_config_t        v_config;          /* config stuff passed to child */
  xine_mrl_t         **mrls;              /* MRLs in media */
  int                  num_mrls;          /* count of above */

  char                *autoplaylist[CDIO_CD_MAX_TRACKS];

  char                *cdda_device;              /* Device name to use when
						    none specified in MRL
						  */
  bool                need_init;
  CdIo *              cdio;           /* libcdio uses this to read          */
#ifdef HAVE_CDDB
  struct  {
    bool             enabled;        /* Do we use CDDB at all? */
    char            *server;         /* CDDB Server to contact */
    int              port;           /* Port number for server above */
    bool             http;           /* Contact using HTTP protocol? */
    char            *email;          /* Email we report back to CDDB. Can
					be bogus to hide information. 	   */
#if CDDB_HAD_X_CONFIGURABLE_CACHEDIR  
    char            *cachedir;
#endif
  } cddb;
#endif


};


/**************** Static prototypes **************/

static bool  cdda_handle_events(cdda_input_plugin_t *this);
static bool  cdda_play_track (cdda_input_plugin_t *this, track_t track_num);
static void  cdda_update_title(cdda_input_plugin_t *this);
static off_t cdda_plugin_get_current_pos (input_plugin_t *this_gen);

/*
 * Config callbacks
 */

static void 
cdda_debug_cb(void *this_gen, xine_cfg_entry_t *entry) 
{
  dbg_print(INPUT_DBG_CALL, "called setting %d\n", entry->num_value);
  cdda_debug = entry->num_value; 
}

#define IP_CALLBACK(fn_name, var, field)				\
  static void fn_name(void *data, xine_cfg_entry_t *cfg)		\
  {									\
    cdda_input_class_t *class = (cdda_input_class_t *) data;		\
    dbg_print(INPUT_DBG_CALL, "called setting %d\n", cfg->num_value);   \
									\
    if (class->ip) {							\
      cdda_input_plugin_t *this = class->ip;				\
      this->var = cfg->field;						\
    }									\
  }									
    
#define CHANGED_CALLBACK(fn_name, var, field)				\
  static void fn_name(void *data, xine_cfg_entry_t *cfg)		\
  {									\
    cdda_input_class_t *class = (cdda_input_class_t *) data;		\
    dbg_print(INPUT_DBG_CALL, "called setting %d\n", cfg->num_value);   \
									\
    class->var = cfg->field;						\
  }									
    
CHANGED_CALLBACK(cdda_device_changed_cb,   cdda_device,   str_value);

IP_CALLBACK(cddb_disk_mode_changed_cb,    v_config.disk_mode,  
	    num_value);
IP_CALLBACK(cddb_title_format_changed_cb,  v_config.title_format,  str_value);
IP_CALLBACK(cddb_comment_format_changed_cb,v_config.comment_format,str_value);


#ifdef HAVE_CDDB
IP_CALLBACK(cddb_cddb_title_format_changed_cb, v_config.cddb_title_format,  
	    str_value);
IP_CALLBACK(cddb_cddb_comment_format_changed_cb,v_config.cddb_comment_format,
	    str_value);

CHANGED_CALLBACK(cddb_enabled_changed_cb,  cddb.enabled,  num_value);
CHANGED_CALLBACK(cddb_server_changed_cb,   cddb.server,   str_value);
CHANGED_CALLBACK(cddb_port_changed_cb,     cddb.port,     num_value);
CHANGED_CALLBACK(cddb_http_changed_cb,     cddb.http,     num_value);
#if CDDB_HAD_CONFIGURABLE_CACHEDIR  
CHANGED_CALLBACK(cddb_cachedir_changed_cb, cddb.cachedir, str_value);
#endif
CHANGED_CALLBACK(cddb_email_changed_cb,    cddb.email,    str_value);

/*
 * **************** CDDB *********************
 */


#if FIXED
/*
 * Return 1 if CD has been changed, 0 of not, -1 on error.
 * FIXME! This gets moved somewhere else!
 */
static int _cdda_is_cd_changed(cdda_input_plugin_t *this) {
#ifdef CDROM_MEDIA_CHANGED
  int err, cd_changed=0;
  
  if(this == NULL || this->fd < 0)
    return -1;
  
  if((err = ioctl(this->fd, CDROM_MEDIA_CHANGED, cd_changed)) < 0) {
    printf("input_cdda: ioctl(CDROM_MEDIA_CHANGED) failed: %s.\n", 
	   strerror(errno));
    return -1;
  }
  
  switch(err) {
  case 1:
    return 1;
    break;
    
  default:
    return 0;
    break;
  }

  return -1;
#else
  /*
   * At least on solaris, CDROM_MEDIA_CHANGED does not exist. Just return an
   * error for now 
   */
  return -1;
#endif
}
#endif /* FIXED */

#if CDDB_HAD_CONFIGURABLE_CACHEDIR  
/*
 * Where, by default, cddb cache files will be saved
 */
static char *_cdda_cddb_get_default_location(void) {
  static char buf[XINE_PATH_MAX + XINE_NAME_MAX + 1];
  
  memset(&buf, 0, sizeof(buf));
  sprintf(buf, "%s/.cddbslave", (xine_get_homedir()));
  
  return buf;
}
#endif

#define free_and_dup(var, val) \
  if (var) free(var);	       \
  if (val) var=strdup(val);	       
  

/*
 * Set information from CDDB
 */
static void 
_cdda_cddb_set_info(cdda_input_plugin_t *this, cddb_disc_t *disc) {

  if (this == NULL) return;

  free_and_dup(this->cddb.disc_title,  disc->title);
  free_and_dup(this->cddb.disc_artist, disc->artist);
  free_and_dup(this->cddb.disc_genre,  disc->genre);

  this->cddb.disc_category = disc->category;
  this->cddb.disc_seconds  = disc->length;
  this->cddb.disc_id       = disc->discid;

  if (disc->year != 0)
    snprintf(this->cddb.disc_year, 5, "%d", disc->year);

  this->cddb.have_info = disc->title != NULL || disc->artist != NULL
    || disc->genre != NULL || disc->year != 0;

}

/*
 * Free allocated memory for CDDB informations
 */
static void 
_cdda_free_cddb_info(cdda_input_plugin_t *this) {

  if(this->cddb.cdiscid)
    free(this->cddb.cdiscid);
  
  if(this->cddb.disc_title)
    free(this->cddb.disc_title);
  
  if(this->cddb.disc_artist)
    free(this->cddb.disc_artist);
  
  if(this->cddb.disc_genre)
    free(this->cddb.disc_genre);

}
#endif /*CDDB*/

/*
 * **************** MRL-releated *********************
 */

/*! Add another MRL to the MRL list inside "this" to be displayed. 
   mrl is the string name to add; size is the size of the entry in bytes. 
   The number of mrls in "this" is incremented. 
*/
static void
cdda_add_mrl_slot(cdda_input_class_t *this, const char *mrl, off_t size, 
		  track_t i)
{
  dbg_print(INPUT_DBG_MRL, "called to add slot %d: %s, size %u\n", 
              i, mrl, (unsigned int) size);
  
  this->mrls[i] = malloc(sizeof(xine_mrl_t));
  if (NULL==this->mrls[i]) {
    LOG_ERR("Can't malloc %d bytes for MRL slot %d (%s)", 
            sizeof(xine_mrl_t), i, mrl);
    return;
  }
  this->mrls[i]->link   = NULL;
  this->mrls[i]->origin = NULL;
  this->mrls[i]->type   = mrl_cda;
  this->mrls[i]->size   = size * CDIO_CD_FRAMESIZE_RAW;
  
  this->mrls[i]->mrl = (char *) malloc(strlen(mrl) + 1);
  if (NULL==this->mrls[i]->mrl) {
    LOG_ERR("Can't malloc %d bytes for MRL name %s", sizeof(xine_mrl_t), mrl);
  } else {
    sprintf(this->mrls[i]->mrl, "%s", mrl);
  }
}

static bool
cdda_build_mrl_list(cdda_input_class_t *class)
{

  char        *mrl;
  CdIo        *cdio;
  bool         destroy_cdio = false;
  track_t      first_track, i;
  track_t      num_tracks;
  unsigned int size;

  if (!class->need_init && class->cdio) {
    cdio = class->cdio;
  } else {
    cdio = cdio_open(class->cdda_device, DRIVER_UNKNOWN);
    if (NULL == cdio) {
      LOG_MSG("%s", _("cdio open failed"));
      return false;
    }
    destroy_cdio = true;
  }
  
  num_tracks = cdio_get_num_tracks(cdio);
  if (CDIO_INVALID_TRACK == num_tracks) {
    LOG_MSG("%s", _("error retrieving number of tracks"));
    return false;
  }
  
  xine_free_mrls(&(class->num_mrls), class->mrls);

  class->num_mrls = num_tracks;
  class->mrls     = calloc(class->num_mrls, sizeof(xine_mrl_t *));

  if (NULL == class->mrls) {
    LOG_ERR("Can't calloc %d MRL entries", class->num_mrls);
    class->num_mrls = 0;
    return false;
  }


  size = MRL_PREFIX_LEN+strlen(class->cdda_device)+6;
  mrl = (char *) malloc(size);
  if (NULL==mrl) {
    LOG_ERR("Can't malloc %d bytes for MRL", size);
    return false;
  }

  /* Record MRL's for tracks */
  first_track = cdio_get_first_track_num(cdio);
  for ( i= 0; i < class->num_mrls; i++) { 
    memset(mrl, 0, sizeof (mrl));
    MRL_FORMAT(mrl, class->cdda_device, first_track+i);
    cdda_add_mrl_slot(class, mrl, 
		      cdio_get_track_sec_count(cdio, first_track+i), i);
  }

  free(mrl);
  if (destroy_cdio) cdio_destroy(cdio);
    
  return true;
}

/*!
  parses a MRL which has the format
    cdda:/[cd_path][:number]?

  Parameters:
    mrl            the MRL we need to parse
    default_device is the device we fill in if one isn't specified in the
                   MRL
    device_str     we set this to the device implied on success
    track_num      we set this to the track number on success
  
  Return true if we have a valid MRL and false if we don't.

  Examples of valid MRL's
    cdda:                     - track 1 of device: /dev/cdrom
    cdda:/                    - same as above
    cdda:/:                   - same as above
    cdda://dev/cdrom          - probably same as above
    cdda://dev/cdrom2         - track 1 of /dev/cdrom2
    cdda://dev/cdrom2:        - same as above
    cdda://dev/cdrom2:5       - track 5 from /dev/cdrom2
    cdda:/:2                  - track 2 from default device
    cdda:3                    - track 3 from default device
    cdda://dev/cdrom2:4       - track 4 from /dev/cdrom2
    cdda://tmp/ntsc.cue       - track 1 from /tmp/ntsc.bin, (a bin/cue
                                disk image)
    cdda://tmp/ntsc.cue:      - same as above
    cdda:/ntsc.nrg            - track 1 of ntsc.nrg (a nero disk image)
    cdda://tmp/ntsc.nrg:5     - track 5 of /tmp/ntsc.nrg

  Bad MRL's
    cdda:/:x                  - x is not a number
    cdda/tmp                  - no colon
    cdda://:tmp               - again :tmp is not a number

 */

static bool
cdda_parse_mrl(/*in*/ const char *default_device, /*in*/ char *mrl, 
	       /*out*/ char *device_str, /*out*/ track_t *track_num)
{
  int count;
  char *p;
  unsigned int num = 0;

  if ( NULL != mrl && !strncasecmp(mrl, MRL_PREFIX, MRL_PREFIX_LEN) )
    p = &mrl[MRL_PREFIX_LEN];
  else
    return false;

  if (0 == strlen(p)) {
    *track_num = 1;
    strcpy(device_str, default_device);
    return true;
  }
  
  if ('/' == p[0]) p++;

  /* try to match device (part after cdda: and before next colon) 
     and track number (integer after colon).
   */
  count = sscanf (p, "%[^:]:%u", device_str, &num);
  
  switch (count) {
  case 2:
    /* Got both device and track. device was set above. */
    *track_num = num;
    return true;
  case 1: 
    {
      /* Matched device, but nothing beyond that */
      
      int len_p = strlen(p);
      int len_d = strlen(device_str);

      /* If there was a colon but more stuff after that, then fail.
	 A single colon at the end is okay though.
       */
      if ((len_p==len_d+1) && (':' != p[len_d])) 
	return false;

      if (len_p != len_d && len_p != len_d+1)
	return false;

      /* See if we have old-style MRL, e.g. cdda:n */
      count = sscanf (device_str, "%u", &num);
      if (1==count) {
	*track_num = num;
	strcpy(device_str, default_device);
	return true;
      } else if (0==count) {
	/* Have device, so just take default track 1. */
	*track_num = 1;
	return true;
      } else 
	return false;
    }
    
  case 0:
  case EOF: 
    {
      off_t default_device_len = strlen(default_device);
      /* No device/file given, so use the default device and try again. */
      strncpy(device_str, default_device, default_device_len+1);
      if (p[0] == ':') p++;
      count = sscanf (p, "%u", &num);
      
      switch (count) {
      case EOF:
	*track_num = 1;
        return true;
      case 1:
        *track_num = num;
	return true;
      default:
	return false;
      }
    }
  }
  return false;
}

static char* 
cdda_plugin_get_mrl (input_plugin_t *this_gen) {
  cdda_input_plugin_t *this = (cdda_input_plugin_t *) this_gen;

  return this->mrl;
}

/*!
  From xine plugin spec:

  generate autoplay list
  return value: list of MRLs

-- The list of MRLs returned goes into the playlist.
   This is called when the SHORT_PLUGIN_NAME button is pressed.
*/

static char ** 
cdda_class_get_autoplay_list (input_class_t *this_gen, int *num_files) {

  cdda_input_class_t *class = (cdda_input_class_t *) this_gen;
  char track_mrl[200];
  int i;
  track_t first_track;
  CdIo *cdio;

  dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), "called\n");

  /* free old playlist */
  for( i = 0; class->autoplaylist[i]; i++ ) {
    free( class->autoplaylist[i] );
    class->autoplaylist[i] = NULL; 
  }  
  
  cdio = cdio_open(class->cdda_device, DRIVER_UNKNOWN);
  if (NULL == cdio) {
    LOG_MSG("%s", _("cdio open failed"));
    *num_files = 0;
    return NULL;
  }

  if ( !cdda_build_mrl_list(class) ) {
    *num_files = 0;
    return NULL;
  }
  
  first_track = cdio_get_first_track_num(cdio);
  *num_files = cdio_get_num_tracks(cdio);
  if (CDIO_INVALID_TRACK == *num_files) {
    LOG_MSG("%s", _("error retrieving number of tracks"));
    cdio_destroy(cdio);
    return NULL;
  }
  for( i = 0; i <= *num_files; i++ ) {
    /* We only want to process audio CDs. */
    if (cdio_get_track_format(cdio, first_track+i) != TRACK_FORMAT_AUDIO) {
      LOG_MSG("track %d is not an audio track", first_track+i);
    } else {
      MRL_FORMAT(track_mrl, class->cdda_device, first_track+i);
      class->autoplaylist[i] = strdup(track_mrl);    
    }
  }
  cdio_destroy(cdio);

  return class->autoplaylist;
}

/*
 * **************** LOGGING  *********************
 */

/* Pointer to cdio default log handler. Set by init_input_plugin
   routine. Perhaps can remove. */
static cdio_log_handler_t gl_default_cdio_log_handler = NULL;

/*! This routine is called by vcd routines on error. 
   Setup is done by init_input_plugin.
*/
static void 
cdda_log_handler (cdio_log_level_t level, const char message[])
{
  switch (level) {
  case CDIO_LOG_DEBUG:
  case CDIO_LOG_INFO:
    if (!(cdda_debug & INPUT_DBG_INFO))
      return;
    /* Fall through if to warn case */
  case CDIO_LOG_WARN:
    LOG_MSG("%s", message);
    break;
  case CDIO_LOG_ERROR:
  case CDIO_LOG_ASSERT:
    LOG_ERR("%s", message);
    break;
  default:
    LOG_ERR("%s\n%s %d", 
            message, 
            _("The above message had unknown cdio log level"), 
            level);
  }
  
  /* gl_default_cdio_log_handler (level, message); */
}

/*! This routine is called by libcdio routines on error. 
   Setup is done by init_input_plugin.
*/
static void 
cdio_log_handler (cdio_log_level_t level, const char message[])
{
  switch (level) {
  case CDIO_LOG_DEBUG:
  case CDIO_LOG_INFO:
    if (!(cdda_debug & INPUT_DBG_CDIO)) return;
    /* Fall through if to warn case */
  default:
    cdda_log_handler (level, message);
  }
}

/*! This routine is when xine is not around. 
   Setup is done by cdio_class_dispose.
*/
static void 
uninit_log_handler (cdio_log_level_t level, const char message[])
{
  switch (level) {
  case CDIO_LOG_DEBUG:
  case CDIO_LOG_INFO:
    if (!(cdda_debug & (INPUT_DBG_INFO|INPUT_DBG_CDIO)))
      return;
    /* Fall through if to warn case */
  case CDIO_LOG_WARN:
    fprintf(stderr, "WARN: %s\n", message);
    break;
  case CDIO_LOG_ERROR:
    fprintf(stderr, "ERROR: %s\n", message);
    break;
  case CDIO_LOG_ASSERT:
    fprintf(stderr, "ASSERT ERROR: %s\n", message);
    break;
  default:
    fprintf(stderr, "UNKNOWN ERROR: %s\n%s %d\n",
            message, 
            _("The above message had unknown cdio log level"), 
            level);
  }
  
  /* gl_default_cdio_log_handler (level, message); */
}

static uint32_t 
cdda_plugin_get_capabilities (input_plugin_t *this_gen) {

  uint32_t ret = INPUT_CAP_SEEKABLE | INPUT_CAP_BLOCK | INPUT_CAP_CHAPTERS;
  dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), "returning %d\n", ret);
  cdda_handle_events((cdda_input_plugin_t *) this_gen);
  return ret;
}


static off_t 
cdda_plugin_read (input_plugin_t *this_gen, char *buf, off_t len) {

  /* only allow reading in block-sized chunks */

  return 0;
}

/* 44100 samples/sec * 2 bytes/samples * 2 channels */
#define CD_BYTES_PER_SECOND (44100 * 2 * 2)

static buf_element_t *
cdda_plugin_read_block (input_plugin_t *this_gen, fifo_buffer_t *fifo, 
  off_t nlen) {

  cdda_input_plugin_t *this = (cdda_input_plugin_t *) this_gen;
  buf_element_t *buf;
  unsigned char frame_data[CDIO_CD_FRAMESIZE_RAW];

  dbg_print((INPUT_DBG_CALL), "called\n");

  if (nlen != CDIO_CD_FRAMESIZE_RAW)
    return NULL;

  cdda_handle_events(this);

  if (this->current_frame >= this->last_frame) {
    dbg_print( INPUT_DBG_LBA, 
              "end reached, cur: %u, end: %u\n", this->current_frame, 
	       this->last_frame);
    if (this->v_config.disk_mode==track_mode) goto end_of_cd;
    if (this->cur_track - this->first_track < this->num_tracks-1) 
      cdda_play_track(this, this->cur_track+1);
    else 
      goto end_of_cd;
  }

  if (cdio_read_audio_sector(this->class->cdio, frame_data, 
			     this->current_frame++)) {
    return NULL;
  }

  buf = fifo->buffer_pool_alloc(fifo);
  buf->content = buf->mem;
  buf->type = BUF_DEMUX_BLOCK;
  buf->size = CDIO_CD_FRAMESIZE_RAW;
  memcpy(buf->mem, frame_data, CDIO_CD_FRAMESIZE_RAW);

  if (this->jumped) {
    int64_t pts = cdda_plugin_get_current_pos(this_gen);
    pts = (pts * 90000) / CD_BYTES_PER_SECOND;
    xine_demux_control_newpts(this->stream, pts, 0);
    this->jumped = false;
  }

  return buf;
 end_of_cd: 
  this->class->need_init = true;
  return NULL;
}

/*
  Set up internal state so that we play a given track.
 */
static bool
cdda_find_track (cdda_input_plugin_t *this, lsn_t frame) 
{
  track_t t;
  lsn_t last_lsn, this_lsn;
  
  dbg_print((INPUT_DBG_CALL), "called frame: %d\n", frame);

  /* Catch the easy and common case where there's no searching to do
     because we stay inside the track.
   */
  if (frame >= this->first_frame && frame <= this->last_frame)
    return true;

  last_lsn = cdio_get_track_lsn(this->class->cdio, this->first_track);
  for (t=1; t<this->num_tracks; t++) {
    this_lsn = cdio_get_track_lsn(this->class->cdio, t+this->first_track);
    if ( frame < this_lsn ) {
      this->first_frame = last_lsn;
      this->last_frame  = this_lsn;
      this->cur_track   = t+this->first_track-1;
      this->mrl         = this->class->mrls[t]->mrl;
      cdda_update_title(this);
      return true;
    }
    last_lsn = cdio_get_track_lsn(this->class->cdio, t+this->first_track);
  }

  return false;
}

static off_t 
cdda_plugin_seek (input_plugin_t *this_gen, off_t offset, int origin) {
  cdda_input_plugin_t *this = (cdda_input_plugin_t *) this_gen;
  long int seek_to_frame;
  const long int offset_frames = offset / CDIO_CD_FRAMESIZE_RAW;
  lsn_t first_frame, last_frame;
  

  dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), "called: %ld %d\n", 
	    (long int) offset, origin);

  if (this->v_config.disk_mode == track_mode) {
    first_frame = this->first_frame;
    last_frame  = this->last_frame;
  } else {
    first_frame = 0;
    last_frame  = this->last_disk_frame;
  }
  
  /* compute the proposed frame and check if it is within bounds */
  if (origin == SEEK_SET)
    /* Set absolute. Well relative to track or beginning of disk... */
    seek_to_frame = this->first_frame + offset_frames;
  else if (origin == SEEK_CUR)
    /* Set relative to current position. */
    seek_to_frame = this->current_frame + offset_frames;
  else
    /* Set to be an invalid frame */
    seek_to_frame = last_frame + 1;
  
  if ((seek_to_frame >= first_frame) &&
      (seek_to_frame <= last_frame)) {
    this->current_frame = seek_to_frame;
    cdda_find_track(this, seek_to_frame);
  }
  
  return (this->current_frame-1 - first_frame) * CDIO_CD_FRAMESIZE_RAW;
}

static off_t 
cdda_plugin_get_current_pos (input_plugin_t *this_gen){
  cdda_input_plugin_t *this = (cdda_input_plugin_t *) this_gen;
  off_t frame_offset = (this->v_config.disk_mode == disk_mode)
    ? this->current_frame
    : this->current_frame - this->first_frame;

  return (frame_offset-1) * CDIO_CD_FRAMESIZE_RAW;
}


/* The get_length routine is called a bit. Make reasonably fast by
   caching the last value which doesn't change all that much.
 */

static off_t cdda_plugin_get_length (input_plugin_t *this_gen) {
  cdda_input_plugin_t *this = (cdda_input_plugin_t *) this_gen;

  cdda_handle_events(this);
  if (this->v_config.disk_mode == disk_mode) return this->disk_size;
  return  (this->last_frame - this->first_frame + 1) * CDIO_CD_FRAMESIZE_RAW;
}

static uint32_t cdda_plugin_get_blocksize (input_plugin_t *this_gen) {

  return CDIO_CD_FRAMESIZE_RAW;
}


/* 
   Handle a keyboard/mouse event. Return TRUE if this causes a change
   in the play item.
*/

static bool
cdda_handle_events (cdda_input_plugin_t *this) 
{
  xine_event_t *event;
  int digit_entered=0;

  /* What you add to the last input number entry. It accumulates all of
     the 10_ADD keypresses. This can be turned negative if we are going
     backwards.
  */
  static int number_addend = 0; 

  while ((event = xine_event_get(this->event_queue))) {

    dbg_print( (INPUT_DBG_CALL|INPUT_DBG_EVENT), 
               "processing %d\n", event->type );
    digit_entered=0;

    switch(event->type) {

    case XINE_EVENT_INPUT_NUMBER_10_ADD:
      number_addend += 10; 
      dbg_print(INPUT_DBG_EVENT, "10 added to number. Is now: %d\n", 
                number_addend);
      break;

    /* The method used below is oblivious to XINE_EVENT_INPUT encodings
       In particular, it does not assume XINE_EVENT_INPUT_NUMBE_9 = 
       XINE_EVENT_INPUT_NUMBER_0 + 9. 
     */
    case XINE_EVENT_INPUT_NUMBER_9:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_8:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_7:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_6:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_5:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_4:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_3:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_2:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_1:
      digit_entered++;
    case XINE_EVENT_INPUT_NUMBER_0:
      {
        number_addend *= 10;
        number_addend += digit_entered;
        dbg_print(INPUT_DBG_EVENT, 
                  "digit added number is now: %d\n", number_addend);
        break;
      }
    case XINE_EVENT_INPUT_PREVIOUS:
      number_addend = (number_addend == 0) ? -1 : -number_addend;
      
      /* Fall through below.. */
    case XINE_EVENT_INPUT_NEXT: 
      {
        int num = number_addend;
        
        number_addend = 0;
        
        /* If no number was given it's really the same as 1, not 0. */
        if (num == 0) num++;
        
        dbg_print(INPUT_DBG_EVENT, "NEXT/PREV iteration count %d\n", num);
	
	return cdda_play_track(this, this->cur_track+num);
        break;
      }
    case XINE_EVENT_INPUT_SELECT:
      {
        /* In the future will have to test to see if we are in a menu
           selection. But if not... */
        int num = number_addend;
        number_addend = 0;
        
        /* If no number was given it's really the same as 1, not 0. */
        if (num == 0) num++;

	return cdda_play_track(this, num);
        break;
      }

    }
  }
  return false;
}

static int 
cdda_plugin_get_optional_data (input_plugin_t *this_gen, void *data, 
			       int data_type) {
  return INPUT_OPTIONAL_UNSUPPORTED;
}

static void 
cdda_plugin_dispose (input_plugin_t *this_gen ) {
  cdda_input_plugin_t *this = (cdda_input_plugin_t *) this_gen;

  gl_default_cdio_log_handler = cdio_log_set_handler (uninit_log_handler);

#ifdef HAVE_CDDB
  _cdda_free_cddb_info(this);
  cddb_disc_destroy(this->cddb.disc);
#endif

  free(this->mrl);
  free(this);
}

#define add_format_str_info(val)			\
  {							\
    const char *str = val;				\
    unsigned int len;					\
    if (val != NULL) {					\
      len=strlen(str);					\
      if (len != 0) {					\
	strncat(tp, str, TEMP_STR_LEN-(tp-temp_str));	\
	tp += len;					\
      }							\
      saw_control_prefix = false;			\
    }							\
  }

#define add_format_num_info(val, fmt)			\
  {							\
    char num_str[10];					\
    unsigned int len;                                   \
    sprintf(num_str, fmt, val);				\
    len=strlen(num_str);                                \
    if (len != 0) {					\
      strncat(tp, num_str, TEMP_STR_LEN-(tp-temp_str));	\
      tp += len;					\
    }							\
    saw_control_prefix = false;				\
  }

/*!
   Take a format string and expand escape sequences, that is sequences that
   begin with %, with information from the current CD. 
   The expanded string is returned. Here is a list of escape sequences:

   %a : The artist
   %A : The album information 
   %C : Category
   %I : CDDB disk ID
   %G : Genre
   %M : The current MRL
   %T : The track number
   %t : The name
   %Y : The year 19xx or 20xx
   %% : a %
*/
static char *
cdda_format_str(cdda_input_plugin_t *this, const char format_str[])
{
#define TEMP_STR_SIZE 256
#define TEMP_STR_LEN (TEMP_STR_SIZE-1)
  static char    temp_str[TEMP_STR_SIZE];
  size_t i;
  char * tp = temp_str;
  bool saw_control_prefix = false;
  size_t format_len = strlen(format_str);

  bzero(temp_str, TEMP_STR_SIZE);

  for (i=0; i<format_len; i++) {

    if (!saw_control_prefix && format_str[i] != '%') {
      *tp++ = format_str[i];
      saw_control_prefix = false;
      continue;
    }

    switch(format_str[i]) {
    case '%':
      if (saw_control_prefix) {
	*tp++ = '%';
      }
      saw_control_prefix = !saw_control_prefix;
      break;
#ifdef HAVE_CDDB      
    case 'a':
      if (!this->class->cddb.enabled) goto not_special;
      add_format_str_info(this->cddb.disc_artist);
      break;
    case 'A':
      if (!this->class->cddb.enabled) goto not_special;
      add_format_str_info(this->cddb.disc_title);
      break;
    case 'C':
      if (!this->class->cddb.enabled) goto not_special;
      add_format_str_info(CDDB_CATEGORY[this->cddb.disc_category]);
      break;
    case 'G':
      if (!this->class->cddb.enabled) goto not_special;
      add_format_str_info(this->cddb.disc_genre);
      break;
    case 'I':
      if (!this->class->cddb.enabled) goto not_special;
      add_format_num_info(this->cddb.disc_id, "%x");
      break;
    case 'Y':
      if (!this->class->cddb.enabled) goto not_special;
      add_format_str_info(this->cddb.disc_year);
      break;
    case 't':
      if (this->class->cddb.enabled) {
	cddb_track_t *t=cddb_disc_get_track(this->cddb.disc, 
					    this->cur_track-1);
	if (t != NULL && t->title != NULL) 
	  add_format_str_info(t->title);
      } else goto not_special;
      break;

#endif

    case 'M':
      add_format_str_info(this->mrl);
      break;
    case 'T':
      add_format_num_info(this->cur_track, "%d");
      break;
#ifdef HAVE_CDDB      
    not_special:
#endif
    default:
      *tp++ = '%'; 
      *tp++ = format_str[i];
      saw_control_prefix = false;
    }
  }
  return strdup(temp_str);
}


static void 
meta_info_assign(int field, xine_stream_t *stream, const char * info) 
{
  if (NULL != info) {
    if (stream->meta_info[field])
      free(stream->meta_info[field]);
    dbg_print(INPUT_DBG_META, "meta[%d]: %s\n", field, info);
    stream->meta_info[field] = strdup(info);
  }
}

/* Update the xine player title text. */
static void 
cdda_update_title(cdda_input_plugin_t *this) 
{
  xine_event_t uevent;
  xine_ui_data_t data;

  char *title_str, *title_format, *comment_format;

  if (this->stream == NULL) return;

#ifdef HAVE_CDDB
  if (this->cddb.have_info) {
    title_format   = this->v_config.cddb_title_format;
    comment_format = this->v_config.cddb_comment_format;
  } else 
#endif
    {
      title_format   = this->v_config.title_format;
      comment_format = this->v_config.comment_format;
    }
  
  title_str          = cdda_format_str(this, title_format);
  comment_format     = cdda_format_str(this, comment_format);

  meta_info_assign(XINE_META_INFO_TITLE,   this->stream, title_str);
  meta_info_assign(XINE_META_INFO_COMMENT, this->stream, comment_format);

  /* Set title/chapter display */
  dbg_print((INPUT_DBG_MRL|INPUT_DBG_CALL),
            "Changing title to read '%s'\n", title_str);
  uevent.type        = XINE_EVENT_UI_SET_TITLE;
  uevent.stream      = this->stream;
  uevent.data        = &data;
  uevent.data_length = sizeof(data);

  memcpy(data.str, title_str, strlen(title_str) + 1);
  data.str_len = strlen(title_str) + 1;

  xine_event_send(this->stream, &uevent);
}

/*
  Set up internal state so that we play a given track.
 */
static bool
cdda_play_track (cdda_input_plugin_t *this, track_t track_num) 
{
  track_t t = track_num - this->first_track;
  
  dbg_print((INPUT_DBG_CALL), "called track: %d\n", track_num);

  if (track_num > this->num_tracks) {
    LOG_ERR("CD has %d tracks, and you requested track %d", 
	    this->num_tracks, track_num);
    return false;
  }

  this->cur_track = track_num;

  if (this->class->num_mrls > t) 
    this->mrl = this->class->mrls[t]->mrl;

  /* set up the frame boundaries for this particular track */
  this->first_frame = this->current_frame = 
    cdio_get_track_lsn(this->class->cdio, track_num);
  this->last_frame  = cdio_get_track_lsn(this->class->cdio, track_num+1) - 1;
  this->jumped      = true;

  /*xine_demux_flush_engine (this->stream);*/
  cdda_update_title(this);
  
  return true;
}

/* 
   No special initialization needed here. All of the initialization 
   is either done in the class or when we have an actual MRL we want
   to deal with.
*/
static int 
cdda_plugin_open (input_plugin_t *this_gen ) {
  return 1;
}

/* 
   Basically sets up stream specified by MRL for playing. After this
   routine is called, xine-lib can read blocks from the thing
   specified by the MRL, set the position of the thing specified by the
   MRL, get its size or read its current position...
*/
static input_plugin_t *
cdda_class_get_instance (input_class_t *cls_gen, xine_stream_t *stream, 
			 const char *mrl) {

  cdda_input_plugin_t *this;
  cdda_input_class_t  *class = (cdda_input_class_t *) cls_gen;
  track_t              track_num;

#if CDDB_HAD_CONFIGURABLE_CACHEDIR  
  xine_cfg_entry_t     cachedir_entry;
#endif

  CdIo                *cdio = NULL;
  char                *device_str = 
    (char *) malloc(strlen(mrl)+strlen(class->cdda_device)+1);
  char                *my_mrl = strdup(mrl) ;

  dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), "called %s\n", mrl);

  if (!cdda_parse_mrl(class->cdda_device, my_mrl, device_str, &track_num)) {
    dbg_print(INPUT_DBG_MRL, "parsing MRL %s failed", my_mrl);
    goto free_and_return;
  }

  if (NULL == (cdio=cdio_open(device_str, DRIVER_UNKNOWN)) ) {
    LOG_ERR("%s %s", _("Can't open"), device_str);
    goto free_and_return;
  }

  this = (cdda_input_plugin_t *) xine_xmalloc (sizeof (cdda_input_plugin_t));
  this->stream      = stream;
  this->jumped      = false;
  this->event_queue = xine_event_new_queue (stream);
  this->num_tracks  = cdio_get_num_tracks(cdio);
  this->first_track = cdio_get_first_track_num(cdio);
  this->disk_size   = cdio_get_track_lba(cdio, CDIO_CDROM_LEADOUT_TRACK)
    * CDIO_CD_FRAMESIZE_RAW;
  this->last_disk_frame= cdio_get_track_lsn(cdio, CDIO_CDROM_LEADOUT_TRACK);

  class->ip   = this;
  this->class            = class;
  this->class->cdio      = cdio;
  this->class->need_init = false;

  memcpy(&(this->v_config), &(class->v_config), sizeof(this->v_config));
  
  if (class->num_mrls == 0) cdda_build_mrl_list(class);

  if (track_num-this->first_track >= this->num_tracks) {
    LOG_ERR("CD has %d tracks, and you requested track %d", 
	    this->num_tracks, track_num);
    cdio_destroy(cdio);
    free(this);
    goto free_and_return;
  }
  
  if (cdio_get_track_format(cdio, track_num) != TRACK_FORMAT_AUDIO) {
    LOG_ERR("CD track %d is not audio", track_num);
    cdio_destroy(cdio);
    free(this);
    goto free_and_return;
  }
  

#ifdef HAVE_CDDB
  /*
   * CDDB
   */

  if (this->class->cddb.enabled) {
    int i, matches;
    cddb_conn_t  *conn = cddb_new();

    _cdda_free_cddb_info(this);

    if (!conn) {
      LOG_ERR("unable to initialize libcddb");
      goto cddb_destroy;
    }
    
    cddb_set_email_address(conn, this->class->cddb.email);
    cddb_set_server_name(conn, this->class->cddb.server);
    cddb_set_server_port(conn, this->class->cddb.port);
    if (this->class->cddb.http)
      cddb_http_enable(conn);
    else
      cddb_http_disable(conn);
    
    this->cddb.disc = cddb_disc_new();
    if (!this->cddb.disc) {
      LOG_ERR("Unable to create CDDB disc structure.");
      goto cddb_end;
    }
    for(i = 0; i < this->num_tracks; i++) {
      cddb_track_t *t = cddb_track_new(); 
      t->frame_offset = cdio_get_track_lba(cdio, i+1);
      cddb_disc_add_track(this->cddb.disc, t);
    }
    
    this->cddb.disc->length = 
      cdio_get_track_lba(cdio, CDIO_CDROM_LEADOUT_TRACK) 
      / CDIO_CD_FRAMES_PER_SEC;


    if (!cddb_disc_calc_discid(this->cddb.disc)) {
      LOG_ERR("CDDB disc calc failed");
      goto cddb_destroy;
    }

    matches = cddb_query(conn, this->cddb.disc);
    if (matches > 0) {
      if (matches > 1)
	LOG_MSG("Found %d matches in CDDB. Using first one.", matches);
      cddb_read(conn, this->cddb.disc);

      if (cdda_debug & INPUT_DBG_CDDB) 
	cddb_disc_print(this->cddb.disc);

#if FIXED
      if ((_cdda_is_cd_changed(this) == 1)) 
#endif
	_cdda_cddb_set_info(this, this->cddb.disc);
    } else {
      LOG_MSG("CDDB error: %s", cddb_error_str(errno));
    }

  cddb_destroy:
    cddb_destroy(conn);
  }
 cddb_end:

  meta_info_assign(XINE_META_INFO_ALBUM,  this->stream, 
		   this->cddb.disc_title);
  meta_info_assign(XINE_META_INFO_ARTIST, this->stream, 
		   this->cddb.disc_artist);
  meta_info_assign(XINE_META_INFO_GENRE, this->stream, 
		   this->cddb.disc_genre);
  meta_info_assign(XINE_META_INFO_YEAR, this->stream, 
		   this->cddb.disc_year);
    
#endif /*HAVE_CDDB*/

  gl_default_cdio_log_handler = cdio_log_set_handler (cdio_log_handler);

#if INPUT_PLUGIN_IFACE_VERSION != 11
  this->input_plugin.open               = cdda_plugin_open;
#endif
  this->input_plugin.get_capabilities   = cdda_plugin_get_capabilities;
  this->input_plugin.read               = cdda_plugin_read;
  this->input_plugin.read_block         = cdda_plugin_read_block;
  this->input_plugin.seek               = cdda_plugin_seek;
  this->input_plugin.get_current_pos    = cdda_plugin_get_current_pos;
  this->input_plugin.get_length         = cdda_plugin_get_length;
  this->input_plugin.get_blocksize      = cdda_plugin_get_blocksize;
  this->input_plugin.get_mrl            = cdda_plugin_get_mrl;
  this->input_plugin.get_optional_data  = cdda_plugin_get_optional_data;
  this->input_plugin.dispose            = cdda_plugin_dispose;
  this->input_plugin.input_class        = cls_gen;

  this->mrl = strdup(mrl);

  cdda_play_track(this, track_num);

  free (device_str);
  free (my_mrl);
  return &this->input_plugin;

 free_and_return:
  free (device_str);
  free (my_mrl);
  class->need_init = true;
  return NULL;
  
}

static char *
cdda_class_get_identifier (input_class_t *this_gen) {
  return strdup("cdda");
}

static char *
cdda_class_get_description (input_class_t *this_gen) {
  return _(strdup("Compact Disc Digital Audio (also known as CD-DA or Audio CD or CD-DA)"));
}

/*!
  From xine plugin spec:
  ls function
  return value: NULL => filename is a file, **char=> filename is a dir

-- This list returned forms the entries of the GUI MRL "browser".
*/

static xine_mrl_t **
cdda_class_get_dir (input_class_t *this_gen, const char *filename, 
		    int *num_files) {

  cdda_input_class_t *class = (cdda_input_class_t *) this_gen;

  char                intended_cdda_device[1024]="";
  track_t             track_num;

  if (filename == NULL) {
    dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), 
              "called with NULL\n");
    if ( !cdda_build_mrl_list(class) ) {
      *num_files = 0; 
      return NULL; 
    }
  } else {
    char *mrl = strdup(filename);
    dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), 
              "called with %s\n", filename);
    if ( !cdda_parse_mrl(class->cdda_device, mrl, intended_cdda_device, 
			&track_num) ) {
      *num_files = 0; 
      free (mrl);
      return NULL; 
    }
    free (mrl);
  }

  *num_files = class->num_mrls;
  return class->mrls;
}

static void 
cdda_class_dispose (input_class_t *this_gen) {
  cdda_input_class_t  *class = (cdda_input_class_t *) this_gen;

  gl_default_cdio_log_handler = cdio_log_set_handler (uninit_log_handler);

  dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), "called\n");
  free (class->mrls);
  free (class);
}

/* Eject media. If there isn't a problem ejecting, */
static int 
cdda_class_eject_media (input_class_t *this_gen) {
  cdda_input_class_t  *this = (cdda_input_class_t *) this_gen;
  int ret;

  dbg_print((INPUT_DBG_CALL|INPUT_DBG_XINECALL), "called\n");
  if (NULL == this_gen) return 0;
  
  ret = cdio_eject_media(&this->cdio);
  return (ret == 0) || (ret == 2);
}

static void *
cdda_init (xine_t *xine, void *data) {

  cdda_input_class_t  *class;
  config_values_t     *x_conf;
  char                *default_cdda_device = NULL;

  class = (cdda_input_class_t *) xine_xmalloc (sizeof (cdda_input_class_t));

  class->xine                           = xine;
  class->x_conf                         = x_conf = xine->config;
  class->need_init                      = true;

  if (INPUT_PLUGIN_IFACE_VERSION >= 14) {
    LOG_MSG("%s%d\n", 
            _("This plugin should be looked over to see if it is compatible with this input plugin API: "), INPUT_PLUGIN_IFACE_VERSION);
  }

#if INPUT_PLUGIN_IFACE_VERSION == 11
  class->input_class.open_plugin        = cdda_class_get_instance;
#else
  class->input_class.get_instance       = cdda_class_get_instance;
#endif

  class->input_class.get_identifier     = cdda_class_get_identifier;
  class->input_class.get_description    = cdda_class_get_description;
  class->input_class.get_dir            = cdda_class_get_dir;
  class->input_class.get_autoplay_list  = cdda_class_get_autoplay_list;
  class->input_class.dispose            = cdda_class_dispose;
  class->input_class.eject_media        = cdda_class_eject_media;

  default_cdda_device = cdio_get_default_device(NULL);
  if (NULL != default_cdda_device) {
    default_cdda_device = strdup("/dev/cdrom");
  }

  class->mrls        = (xine_mrl_t **) xine_xmalloc(sizeof(xine_mrl_t*));
  class->num_mrls    = 0;
  
  class->cdda_device = x_conf->register_string(x_conf, "cdda.default_device", 
					      default_cdda_device,
					      _("device used for cdda drive"), 
					       NULL, 20, 
					       cdda_device_changed_cb, 
					       (void *) class);
  cdda_debug = 
    x_conf->register_num(x_conf, 
			 "cdda.debug",
			 0,
			 _("debug flag mask"),
			 _("For tracking down bugs in the vcdx plugin."),
			 0, 
			 cdda_debug_cb, 
			 class); 

  class->v_config.disk_mode =
    x_conf->register_enum(x_conf, "cdda.disk_mode", track_mode,
			  (char **) disk_modes,
			  _("play unit"),
_("Do we treat unit of play as disk (with tracks) or play a single "
  "track and stop?"), 
			  0,
			  cddb_disk_mode_changed_cb, (void *) class);
  
#ifdef HAVE_CDDB
  class->cddb.enabled =
    x_conf->register_bool(x_conf, "cdda.cddb_enabled", 1,
			  _("Do we use CDDB to retrieve CD information?"), 
			  NULL, 0,
			  cddb_enabled_changed_cb, (void *) class);
  
  class->cddb.http =
    x_conf->register_bool(x_conf, "cdda.cddb_http", 0,
			  _("Contact CDDB via the HTTP protocol?"), NULL, 0,
			  cddb_http_changed_cb, (void *) class);

  class->cddb.server =
    strdup(x_conf->register_string(x_conf, "cdda.cddb_server", 
				   CDDB_SERVER,
			  _("The server CDDB contacts to get CD info"), 
				   (void *) class, 10,
				   cddb_server_changed_cb, (void *) class));
  class->cddb.port =
    x_conf->register_num(x_conf, "cdda.cddb_port", CDDB_PORT,
			 _("cddb server port"), (void *) class, 10,
			 cddb_port_changed_cb, (void *) class);

#if CDDB_HAD_X_CONFIGURABLE_CACHEDIR  
  class->cddb.cachedir = 
    strdup(x_conf->register_string(x_conf, "cdda.cddb_cachedir", 
				   (_cdda_cddb_get_default_location()),
				   _("cddbp cache directory"), NULL, 20, 
				   cddb_cachedir_changed_cb, (void *) class));
#endif

  class->cddb.email =
    strdup(x_conf->register_string(x_conf, "cdda.cddb_email", 
				   "me@home",
				   _("email given on cddb requests"), 
				   (void *) class, 10, 
				   cddb_email_changed_cb, (void *) class));
#endif

  class->v_config.title_format =
    strdup(x_conf->register_string(x_conf,
				   "cdda.title_format",
				   _("Track %T %M"),
_("format for display banner (when CDDB is not available)"),
"Format used in the GUI Title. Similar to the Unix date "
"command. Format specifiers that start with a percent sign. Specifiers are "
				   "%M, %T %%.",
				   0,
				   cddb_title_format_changed_cb,
				   (void *) class));

  class->v_config.comment_format =
    strdup(x_conf->register_string(x_conf,
				   "cdda.cddb_comment_format",
				   "MRL %M",
_("format for stream comment field (when CDDB info not available)"),
_("Format used in the GUI Title. Similar to the Unix date "
"command. Format specifiers that start with a percent sign. Specifiers are "
"same as the cddb_title_format string."),
				   0,
				   cddb_comment_format_changed_cb,
				   (void *) class));
#ifdef HAVE_CDDB
  class->v_config.cddb_title_format =
    strdup(x_conf->register_string(x_conf,
				   "cdda.cddb_cddb_title_format",
                               _("Track %T: %t Artist: %a, Album %A, %C (%Y)"),
_("format string for display banner (when CDDB info available)"),
"Format used in the GUI Title. Similar to the Unix date "
"command. Format specifiers that start with a percent sign. Specifiers are "
			  "%a, %A, %C, %G, %I %M, %T, %Y and %%.",
				   0,
				   cddb_cddb_title_format_changed_cb,
				   (void *) class));

  class->v_config.cddb_comment_format =
    strdup(x_conf->register_string(x_conf,
				   "cdda.cddb_cddb_comment_format",
				   "MRL %M, Disc Id %I",
				   _("format string for stream comment field"),
_("Format used in the GUI Title. Similar to the Unix date "
"command. Format specifiers that start with a percent sign. Specifiers are "
"same as the cddb_title_format string."),
				   0,
				   cddb_cddb_comment_format_changed_cb,
				   (void *) class));
#endif



  return class;
}

/* 
   Exported plugin catalog entries.

   All plugins listing only the current API number break when the API
   number is increased. This is by design. 

   Sometimes in the rush to get out a buggy release, the API number is
   increased without communication let alone a concern for whether it
   is necessary or how many plugins it might break. And that is
   precisely when what happened between API release 12 and API
   13. Input plugin API numbers 12 and 13 are functionally identical.

   Because of problems like this, we'll just put in a future API
   release. If the number was increased for a reason that doesn't
   affect us (such as for nor reason at all), then this plugin will
   work unmodified that future APIs. If on the other hand there was
   incompatible change, we are no worse off than if we hadn't entered
   the next API number since in both cases the plugin is broken.
 */
const plugin_info_t xine_plugin_info[] = {
  /* type, API, "name", version, special_info, init_function */
  { PLUGIN_INPUT, 11, (char *) SHORT_PLUGIN_NAME, XINE_VERSION_CODE, 
    NULL, cdda_init },
  { PLUGIN_INPUT, 12, (char *) SHORT_PLUGIN_NAME, XINE_VERSION_CODE, 
    NULL, cdda_init },
  { PLUGIN_INPUT, 13, (char *) SHORT_PLUGIN_NAME, XINE_VERSION_CODE, 
    NULL, cdda_init }, /* This is the current API */
  { PLUGIN_INPUT, 14, (char *) SHORT_PLUGIN_NAME, XINE_VERSION_CODE, 
    NULL, cdda_init }, /* This is possibly the next API */
  { PLUGIN_NONE, 0, (char *) "", 0, NULL, NULL }
};
